<?php
/**
 * Webhook Listener
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Webhook Listener Class
 */
class VidToArticle_Webhook_Listener {

	/**
	 * Maximum age of webhook (in seconds)
	 *
	 * @var int
	 */
	const MAX_TIMESTAMP_AGE = 300; // 5 minutes

	/**
	 * Handle incoming webhook
	 *
	 * @param WP_REST_Request $request REST request object.
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_webhook( $request ) {
		// Check idempotency key first (fast path for duplicates).
		$idempotency_key = $request->get_header( 'X-VidToArticle-Idempotency-Key' );
		if ( ! empty( $idempotency_key ) ) {
			$idempotency_key = sanitize_key( $idempotency_key );
			$cached_result   = $this->check_idempotency_cache( $idempotency_key );

			if ( false !== $cached_result ) {
				$this->log_activity(
					'webhook_duplicate_idempotency',
					__( 'Duplicate webhook detected via idempotency key', 'vidtoarticle-publisher' ),
					array(
						'idempotency_key' => $idempotency_key,
						'post_id'         => $cached_result['post_id'],
					)
				);

				return rest_ensure_response( array(
					'success' => true,
					'message' => 'Webhook already processed (idempotent)',
					'post_id' => (int) $cached_result['post_id'],
				) );
			}
		}

		// Get raw body for signature verification.
		$body = $request->get_body();
		$data = json_decode( $body, true );

		if ( null === $data ) {
			return new WP_Error(
				'invalid_json',
				__( 'Invalid JSON payload', 'vidtoarticle-publisher' ),
				array( 'status' => 400 )
			);
		}

		// Verify webhook signature.
		$verification = $this->verify_webhook_signature( $request, $body );
		if ( is_wp_error( $verification ) ) {
			// Get headers for debugging
			$timestamp_header = $request->get_header( 'X-VidToArticle-Timestamp' );
			$signature_header = $request->get_header( 'X-VidToArticle-Signature' );

			$this->log_activity(
				'webhook_rejected',
				$verification->get_error_message(),
				array(
					'error_code' => $verification->get_error_code(),
					'timestamp_header' => $timestamp_header ? 'present' : 'missing',
					'signature_header' => $signature_header ? 'present' : 'missing',
					'timestamp_value' => $timestamp_header ? substr( $timestamp_header, 0, 20 ) : 'N/A',
					'body_size' => strlen( $body ),
					'has_secret_key' => get_option( 'vidtoarticle_secret_key' ) ? 'yes' : 'no',
				)
			);
			return $verification;
		}

		// Extract data.
		$timestamp  = isset( $data['timestamp'] ) ? $data['timestamp'] : '';
		$webhook_data = isset( $data['data'] ) ? $data['data'] : array();

		// Validate required fields.
		if ( empty( $webhook_data['job_id'] ) || empty( $webhook_data['article'] ) ) {
			return new WP_Error(
				'missing_data',
				__( 'Missing required webhook data', 'vidtoarticle-publisher' ),
				array( 'status' => 400 )
			);
		}

		$job_id  = sanitize_text_field( $webhook_data['job_id'] );
		$video   = isset( $webhook_data['video'] ) ? $webhook_data['video'] : array();
		$article = $webhook_data['article'];
		$metadata = isset( $webhook_data['metadata'] ) ? $webhook_data['metadata'] : array();

		// Check if job already processed (prevent duplicates).
		global $wpdb;
		$existing_job = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT id, post_id FROM {$wpdb->prefix}vidtoarticle_jobs WHERE backend_job_id = %s",
				$job_id
			)
		);

		if ( $existing_job && $existing_job->post_id ) {
			// Already processed.
			$this->log_activity( 'webhook_duplicate', __( 'Webhook already processed', 'vidtoarticle-publisher' ), array(
				'job_id'  => $job_id,
				'post_id' => $existing_job->post_id,
			) );

			return rest_ensure_response( array(
				'success' => true,
				'message' => 'Already processed',
				'post_id' => $existing_job->post_id,
			) );
		}

		// Publish the article.
		$publisher = new VidToArticle_Post_Publisher();
		$result    = $publisher->publish_article( $article, $video, $job_id, $metadata );

		if ( is_wp_error( $result ) ) {
			$this->log_activity( 'webhook_publish_failed', $result->get_error_message(), array(
				'job_id' => $job_id,
				'video'  => $video,
			) );

			return new WP_Error(
				'publish_failed',
				$result->get_error_message(),
				array( 'status' => 500 )
			);
		}

		// Send confirmation to backend.
		$api_client = new VidToArticle_API_Client();
		$api_client->confirm_webhook( $job_id );

		// Store idempotency key to prevent future duplicate processing.
		if ( ! empty( $idempotency_key ) ) {
			$this->store_idempotency_cache( $idempotency_key, array(
				'post_id' => $result['post_id'],
				'job_id'  => $job_id,
			) );
		}

		// Log success.
		$this->log_activity( 'webhook_processed', __( 'Article published successfully', 'vidtoarticle-publisher' ), array(
			'job_id'  => $job_id,
			'post_id' => $result['post_id'],
			'title'   => $article['title'],
		) );

		return rest_ensure_response( array(
			'success' => true,
			'message' => 'Article published successfully',
			'post_id' => $result['post_id'],
		) );
	}

	/**
	 * Verify webhook HMAC signature
	 *
	 * @param WP_REST_Request $request REST request object.
	 * @param string          $body    Raw request body.
	 * @return true|WP_Error True if valid, WP_Error otherwise.
	 */
	private function verify_webhook_signature( $request, $body ) {
		// Get headers.
		$timestamp = $request->get_header( 'X-VidToArticle-Timestamp' );
		$signature = $request->get_header( 'X-VidToArticle-Signature' );

		// EXTENSIVE LOGGING FOR DEBUGGING
		error_log( '=== WORDPRESS WEBHOOK SIGNATURE DEBUG ===' );
		error_log( 'Received timestamp: ' . $timestamp );
		error_log( 'Received signature: ' . $signature );
		error_log( 'Body length: ' . strlen( $body ) );
		error_log( 'Body (first 200 chars): ' . substr( $body, 0, 200 ) );

		if ( empty( $timestamp ) || empty( $signature ) ) {
			return new WP_Error(
				'missing_headers',
				__( 'Missing authentication headers', 'vidtoarticle-publisher' ),
				array( 'status' => 401 )
			);
		}

		// Check timestamp (prevent replay attacks).
		$current_time = time();
		$age          = abs( $current_time - intval( $timestamp ) );

		error_log( 'Current time: ' . $current_time );
		error_log( 'Timestamp age: ' . $age . ' seconds' );

		if ( $age > self::MAX_TIMESTAMP_AGE ) {
			return new WP_Error(
				'expired_timestamp',
				__( 'Webhook timestamp too old', 'vidtoarticle-publisher' ),
				array( 'status' => 401 )
			);
		}

		// Get secret key.
		$secret_key = get_option( 'vidtoarticle_secret_key' );

		if ( empty( $secret_key ) ) {
			return new WP_Error(
				'not_connected',
				__( 'Plugin not connected to VidToArticle.com', 'vidtoarticle-publisher' ),
				array( 'status' => 401 )
			);
		}

		error_log( 'Secret key length: ' . strlen( $secret_key ) );
		error_log( 'Secret key (first 10 chars): ' . substr( $secret_key, 0, 10 ) );

		// Calculate expected signature.
		$body_hash         = hash( 'sha256', $body );
		$signature_string  = $timestamp . 'POST' . '/wp-json/vidtoarticle/v1/webhook' . $body_hash;
		$expected_signature = hash_hmac( 'sha256', $signature_string, $secret_key );

		error_log( 'Calculated body hash: ' . $body_hash );
		error_log( 'Signature string: ' . $signature_string );
		error_log( 'Expected signature: ' . $expected_signature );
		error_log( 'Received signature: ' . $signature );
		error_log( 'Signatures match: ' . ( hash_equals( $expected_signature, $signature ) ? 'YES' : 'NO' ) );
		error_log( '=== END WORDPRESS DEBUG ===' );

		// Compare signatures (timing-safe).
		if ( ! hash_equals( $expected_signature, $signature ) ) {
			// Log detailed mismatch info
			$this->log_activity(
				'webhook_signature_mismatch',
				'Signature verification failed',
				array(
					'expected' => $expected_signature,
					'received' => $signature,
					'body_hash' => $body_hash,
					'timestamp' => $timestamp,
					'body_length' => strlen( $body ),
				)
			);

			return new WP_Error(
				'invalid_signature',
				__( 'Invalid webhook signature', 'vidtoarticle-publisher' ),
				array( 'status' => 401 )
			);
		}

		return true;
	}

	/**
	 * Log activity
	 *
	 * @param string $action  Action type.
	 * @param string $message Log message.
	 * @param array  $context Additional context.
	 */
	private function log_activity( $action, $message, $context = array() ) {
		global $wpdb;

		$wpdb->insert(
			$wpdb->prefix . 'vidtoarticle_activity',
			array(
				'action'     => $action,
				'message'    => $message,
				'context'    => ! empty( $context ) ? wp_json_encode( $context ) : null,
				'created_at' => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%s', '%s' )
		);
	}

	/**
	 * Check idempotency cache for duplicate webhook
	 *
	 * @param string $idempotency_key Idempotency key from webhook header.
	 * @return array|false Cached result if found, false otherwise.
	 */
	private function check_idempotency_cache( $idempotency_key ) {
		$transient_key = 'vidtoarticle_idempotency_' . $idempotency_key;
		$cached_data   = get_transient( $transient_key );

		if ( false !== $cached_data && is_array( $cached_data ) ) {
			return $cached_data;
		}

		return false;
	}

	/**
	 * Store idempotency cache to prevent duplicate processing
	 *
	 * Uses WordPress transients with 7-day expiration for automatic cleanup.
	 *
	 * @param string $idempotency_key Idempotency key from webhook header.
	 * @param array  $data            Data to cache (post_id, job_id, etc).
	 * @return bool True if stored successfully.
	 */
	private function store_idempotency_cache( $idempotency_key, $data ) {
		$transient_key = 'vidtoarticle_idempotency_' . $idempotency_key;

		// Store for 7 days (604800 seconds).
		// WordPress transients automatically handle cleanup.
		$stored = set_transient( $transient_key, $data, 7 * DAY_IN_SECONDS );

		if ( $stored ) {
			error_log( 'VidToArticle: Stored idempotency key - ' . $idempotency_key );
		} else {
			error_log( 'VidToArticle: Failed to store idempotency key - ' . $idempotency_key );
		}

		return $stored;
	}
}
