<?php
/**
 * Source Manager
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Source Manager Class
 */
class VidToArticle_Source_Manager {

	/**
	 * Create new source
	 *
	 * @param array $source_data Source configuration.
	 * @return int|WP_Error Source ID or error.
	 */
	public function create_source( $source_data ) {
		global $wpdb;

		// Validate required fields.
		$required = array( 'source_type', 'source_id', 'article_style' );
		foreach ( $required as $field ) {
			if ( empty( $source_data[ $field ] ) ) {
				return new WP_Error( 'missing_field', sprintf(
					/* translators: %s: field name */
					__( 'Missing required field: %s', 'vidtoarticle-publisher' ),
					$field
				) );
			}
		}

		// Check for existing source in local database.
		$existing = $wpdb->get_row( $wpdb->prepare(
			"SELECT id FROM {$wpdb->prefix}vidtoarticle_sources WHERE source_type = %s AND source_id = %s",
			$source_data['source_type'],
			$source_data['source_id']
		) );

		if ( $existing ) {
			return new WP_Error(
				'duplicate_source',
				sprintf(
					/* translators: %s: source type */
					__( 'This %s is already being monitored locally', 'vidtoarticle-publisher' ),
					$source_data['source_type']
				)
			);
		}

		// Sanitize data.
		$data = array(
			'source_type'    => sanitize_text_field( $source_data['source_type'] ),
			'source_id'      => sanitize_text_field( $source_data['source_id'] ),
			'source_name'    => isset( $source_data['source_name'] ) ? sanitize_text_field( $source_data['source_name'] ) : null,
			'article_style'  => sanitize_text_field( $source_data['article_style'] ),
			'wp_category'    => isset( $source_data['wp_category'] ) ? intval( $source_data['wp_category'] ) : null,
			'wp_author'      => isset( $source_data['wp_author'] ) ? intval( $source_data['wp_author'] ) : get_current_user_id(),
			'posts_per_day'  => isset( $source_data['posts_per_day'] ) ? intval( $source_data['posts_per_day'] ) : 1,
			'backfill_count' => isset( $source_data['backfill_count'] ) ? intval( $source_data['backfill_count'] ) : 0,
			'is_active'      => 1,
			'created_at'     => current_time( 'mysql' ),
		);

		// Insert into database.
		$inserted = $wpdb->insert(
			$wpdb->prefix . 'vidtoarticle_sources',
			$data,
			array( '%s', '%s', '%s', '%s', '%d', '%d', '%d', '%d', '%d', '%s' )
		);

		if ( ! $inserted ) {
			return new WP_Error( 'db_error', __( 'Failed to create source', 'vidtoarticle-publisher' ) );
		}

		$source_id = $wpdb->insert_id;

		// Register with backend API.
		$api_client = new VidToArticle_API_Client();
		$response   = $api_client->create_source( array(
			'source_type'    => $data['source_type'],
			'source_id'      => $data['source_id'],
			'article_style'  => $data['article_style'],
			'posts_per_day'  => $data['posts_per_day'],
			'backfill_count' => $data['backfill_count'],
			'settings'       => array(
				'wp_category' => $data['wp_category'],
				'wp_author'   => $data['wp_author'],
			),
		) );

		if ( is_wp_error( $response ) ) {
			// Check if error is due to duplicate source in backend.
			$error_data = $response->get_error_data();
			$is_duplicate = isset( $error_data['status'] ) && $error_data['status'] === 409;

			if ( $is_duplicate ) {
				// Source exists in backend - update local with backend info.
				$existing_source = $response->get_error_data( 'existing_source' );
				if ( $existing_source && isset( $existing_source['source_name'] ) ) {
					$wpdb->update(
						$wpdb->prefix . 'vidtoarticle_sources',
						array( 'source_name' => $existing_source['source_name'] ),
						array( 'id' => $source_id ),
						array( '%s' ),
						array( '%d' )
					);
				}
				// Return success - source now exists in both places.
				return $source_id;
			} else {
				// Delete local source for other errors.
				$wpdb->delete( $wpdb->prefix . 'vidtoarticle_sources', array( 'id' => $source_id ), array( '%d' ) );
				return $response;
			}
		}

		// Store backend source ID.
		if ( isset( $response['data']['source_id'] ) ) {
			$wpdb->update(
				$wpdb->prefix . 'vidtoarticle_sources',
				array( 'source_name' => $response['data']['source_name'] ?? '' ),
				array( 'id' => $source_id ),
				array( '%s' ),
				array( '%d' )
			);
		}

		return $source_id;
	}

	/**
	 * Get all sources
	 *
	 * @return array Sources list.
	 */
	public function get_sources() {
		global $wpdb;

		return $wpdb->get_results(
			"SELECT * FROM {$wpdb->prefix}vidtoarticle_sources ORDER BY created_at DESC"
		);
	}

	/**
	 * Get source by ID
	 *
	 * @param int $source_id Source ID.
	 * @return object|null Source object or null.
	 */
	public function get_source( $source_id ) {
		global $wpdb;

		return $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}vidtoarticle_sources WHERE id = %d",
				$source_id
			)
		);
	}

	/**
	 * Update source
	 *
	 * @param int   $source_id   Source ID.
	 * @param array $source_data Updated data.
	 * @return bool|WP_Error Success or error.
	 */
	public function update_source( $source_id, $source_data ) {
		global $wpdb;

		// Get existing source.
		$source = $this->get_source( $source_id );

		if ( ! $source ) {
			return new WP_Error( 'not_found', __( 'Source not found', 'vidtoarticle-publisher' ) );
		}

		// Prepare update data.
		$data = array();
		$format = array();

		if ( isset( $source_data['article_style'] ) ) {
			$data['article_style'] = sanitize_text_field( $source_data['article_style'] );
			$format[] = '%s';
		}

		if ( isset( $source_data['wp_category'] ) ) {
			$data['wp_category'] = intval( $source_data['wp_category'] );
			$format[] = '%d';
		}

		if ( isset( $source_data['wp_author'] ) ) {
			$data['wp_author'] = intval( $source_data['wp_author'] );
			$format[] = '%d';
		}

		if ( isset( $source_data['posts_per_day'] ) ) {
			$data['posts_per_day'] = intval( $source_data['posts_per_day'] );
			$format[] = '%d';
		}

		if ( isset( $source_data['is_active'] ) ) {
			$data['is_active'] = intval( $source_data['is_active'] );
			$format[] = '%d';
		}

		if ( empty( $data ) ) {
			return true;
		}

		// Update in database.
		$updated = $wpdb->update(
			$wpdb->prefix . 'vidtoarticle_sources',
			$data,
			array( 'id' => $source_id ),
			$format,
			array( '%d' )
		);

		if ( false === $updated ) {
			return new WP_Error( 'db_error', __( 'Failed to update source', 'vidtoarticle-publisher' ) );
		}

		// Update on backend if needed.
		// (Implementation depends on backend API design)

		return true;
	}

	/**
	 * Delete source
	 *
	 * @param int $source_id Source ID.
	 * @return bool|WP_Error Success or error.
	 */
	public function delete_source( $source_id ) {
		global $wpdb;

		// Get source.
		$source = $this->get_source( $source_id );

		if ( ! $source ) {
			return new WP_Error( 'not_found', __( 'Source not found', 'vidtoarticle-publisher' ) );
		}

		// Delete from backend.
		$api_client = new VidToArticle_API_Client();
		$api_client->delete_source( $source->source_id );

		// Delete from local database.
		$deleted = $wpdb->delete(
			$wpdb->prefix . 'vidtoarticle_sources',
			array( 'id' => $source_id ),
			array( '%d' )
		);

		if ( ! $deleted ) {
			return new WP_Error( 'db_error', __( 'Failed to delete source', 'vidtoarticle-publisher' ) );
		}

		return true;
	}

	/**
	 * Parse YouTube URL to extract ID
	 *
	 * @param string $url       YouTube URL.
	 * @param string $type      Type (playlist or channel).
	 * @return string|WP_Error  Extracted ID or error.
	 */
	public function parse_youtube_url( $url, $type = 'playlist' ) {
		// Playlist URL patterns.
		if ( 'playlist' === $type ) {
			if ( preg_match( '/[?&]list=([^&]+)/', $url, $matches ) ) {
				return sanitize_text_field( $matches[1] );
			}
		}

		// Channel URL patterns.
		if ( 'channel' === $type ) {
			// Channel ID.
			if ( preg_match( '/youtube\.com\/channel\/([^\/\?]+)/', $url, $matches ) ) {
				return sanitize_text_field( $matches[1] );
			}
			// Handle (@username) format.
			if ( preg_match( '/youtube\.com\/@([^\/\?]+)/', $url, $matches ) ) {
				return '@' . sanitize_text_field( $matches[1] );
			}
			// Custom URL.
			if ( preg_match( '/youtube\.com\/c\/([^\/\?]+)/', $url, $matches ) ) {
				return sanitize_text_field( $matches[1] );
			}
		}

		return new WP_Error( 'invalid_url', __( 'Invalid YouTube URL', 'vidtoarticle-publisher' ) );
	}
}
