<?php
/**
 * OAuth-Style Connection Handler
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * OAuth Handler Class
 */
class VidToArticle_OAuth_Handler {

	/**
	 * API Base URL
	 *
	 * @var string
	 */
	private static $api_base_url;

	/**
	 * Initialize
	 */
	public static function init() {
		self::$api_base_url = defined( 'VIDTOARTICLE_API_URL' )
			? VIDTOARTICLE_API_URL
			: 'https://api.vidtoarticle.com';

		add_action( 'admin_init', array( __CLASS__, 'handle_oauth_redirect' ) );
	}

	/**
	 * Generate connection URL
	 *
	 * @return string Connection URL.
	 */
	public static function get_connection_url() {
		// Generate one-time token.
		$token = wp_generate_password( 32, false );
		set_transient( 'vidtoarticle_oauth_token', $token, HOUR_IN_SECONDS );

		// Generate state parameter for CSRF protection (OAuth 2.0 best practice).
		$state = wp_generate_password( 32, false );
		set_transient( 'vidtoarticle_oauth_state', $state, HOUR_IN_SECONDS );

		// Get site info.
		$site_url    = get_site_url();
		$webhook_url = rest_url( 'vidtoarticle/v1/webhook' );
		$callback_url = admin_url( 'admin.php?page=vidtoarticle-publisher&oauth_callback=1' );

		// Build connection URL.
		$params = array(
			'site_url'     => $site_url,
			'webhook_url'  => $webhook_url,
			'callback_url' => $callback_url,
			'token'        => $token,
			'state'        => $state,
		);

		return add_query_arg( $params, self::$api_base_url . '/api/wordpress/connect/initiate' );
	}

	/**
	 * Handle OAuth redirect after user authorizes
	 */
	public static function handle_oauth_redirect() {
		// Check if this is an OAuth callback.
		if ( ! isset( $_GET['page'] ) || 'vidtoarticle-publisher' !== $_GET['page'] ) {
			return;
		}

		if ( ! isset( $_GET['oauth_callback'] ) ) {
			return;
		}

		// Check if user is authorized.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action', 'vidtoarticle-publisher' ) );
		}

		// Verify state parameter for CSRF protection (OAuth 2.0 best practice).
		$state = isset( $_GET['state'] ) ? sanitize_text_field( $_GET['state'] ) : '';
		$stored_state = get_transient( 'vidtoarticle_oauth_state' );
		delete_transient( 'vidtoarticle_oauth_state' );

		if ( empty( $state ) || $state !== $stored_state ) {
			wp_die( esc_html__( 'Invalid state parameter. This may be a CSRF attack.', 'vidtoarticle-publisher' ) );
		}

		// Get return token from URL.
		$return_token = isset( $_GET['return_token'] ) ? sanitize_text_field( $_GET['return_token'] ) : '';

		if ( empty( $return_token ) ) {
			// Check for error parameter.
			if ( isset( $_GET['error'] ) ) {
				$error_message = sanitize_text_field( $_GET['error'] );
				self::add_admin_notice( $error_message, 'error' );
				wp_safe_redirect( admin_url( 'admin.php?page=vidtoarticle-publisher' ) );
				exit;
			}

			self::add_admin_notice( __( 'Connection failed: No return token provided', 'vidtoarticle-publisher' ), 'error' );
			wp_safe_redirect( admin_url( 'admin.php?page=vidtoarticle-publisher' ) );
			exit;
		}

		// Verify token matches.
		$stored_token = get_transient( 'vidtoarticle_oauth_token' );
		delete_transient( 'vidtoarticle_oauth_token' );

		if ( $return_token !== $stored_token ) {
			self::add_admin_notice( __( 'Connection failed: Invalid token', 'vidtoarticle-publisher' ), 'error' );
			wp_safe_redirect( admin_url( 'admin.php?page=vidtoarticle-publisher' ) );
			exit;
		}

		// Exchange token for secret key.
		$api_client = new VidToArticle_API_Client();
		$response   = $api_client->exchange_token( $return_token );

		if ( is_wp_error( $response ) ) {
			self::add_admin_notice(
				sprintf(
					/* translators: %s: error message */
					__( 'Connection failed: %s', 'vidtoarticle-publisher' ),
					$response->get_error_message()
				),
				'error'
			);
			wp_safe_redirect( admin_url( 'admin.php?page=vidtoarticle-publisher' ) );
			exit;
		}

		// Save secret key.
		if ( isset( $response['success'] ) && $response['success'] && isset( $response['data']['secret_key'] ) ) {
			$api_client->save_secret_key( $response['data']['secret_key'] );

			// Store additional connection info.
			update_option( 'vidtoarticle_connection_status', array(
				'connected_at' => current_time( 'mysql' ),
				'api_key_id'   => $response['data']['api_key_id'] ?? '',
			) );

			// Log successful connection.
			self::log_activity( 'connection_established', __( 'Successfully connected to VidToArticle.com', 'vidtoarticle-publisher' ) );

			self::add_admin_notice( __( 'Successfully connected to VidToArticle.com!', 'vidtoarticle-publisher' ), 'success' );
		} else {
			self::add_admin_notice( __( 'Connection failed: Invalid response from server', 'vidtoarticle-publisher' ), 'error' );
		}

		// Redirect back to settings page.
		wp_safe_redirect( admin_url( 'admin.php?page=vidtoarticle-publisher' ) );
		exit;
	}

	/**
	 * Disconnect from VidToArticle
	 *
	 * @return bool Success status.
	 */
	public static function disconnect() {
		// Verify nonce.
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'vidtoarticle_disconnect' ) ) {
			return false;
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		// Revoke connection via API.
		$api_client = new VidToArticle_API_Client();
		$response   = $api_client->revoke_connection();

		if ( is_wp_error( $response ) ) {
			self::add_admin_notice(
				sprintf(
					/* translators: %s: error message */
					__( 'Failed to disconnect: %s', 'vidtoarticle-publisher' ),
					$response->get_error_message()
				),
				'error'
			);
			return false;
		}

		// Clear local data.
		delete_option( 'vidtoarticle_connection_status' );

		// Log disconnection.
		self::log_activity( 'connection_revoked', __( 'Disconnected from VidToArticle.com', 'vidtoarticle-publisher' ) );

		self::add_admin_notice( __( 'Successfully disconnected from VidToArticle.com', 'vidtoarticle-publisher' ), 'success' );

		return true;
	}

	/**
	 * Add admin notice
	 *
	 * @param string $message Notice message.
	 * @param string $type    Notice type (success, error, warning, info).
	 */
	private static function add_admin_notice( $message, $type = 'info' ) {
		set_transient( 'vidtoarticle_admin_notice', array(
			'message' => $message,
			'type'    => $type,
		), 30 );
	}

	/**
	 * Log activity
	 *
	 * @param string $action  Action type.
	 * @param string $message Log message.
	 * @param array  $context Additional context.
	 */
	private static function log_activity( $action, $message, $context = array() ) {
		global $wpdb;

		$wpdb->insert(
			$wpdb->prefix . 'vidtoarticle_activity',
			array(
				'action'     => $action,
				'message'    => $message,
				'context'    => ! empty( $context ) ? wp_json_encode( $context ) : null,
				'created_at' => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%s', '%s' )
		);
	}
}

// Initialize.
VidToArticle_OAuth_Handler::init();
