(function () {
  "use strict";

  const { createElement: el, Fragment, useEffect, useState } = wp.element;
  const { Button, Notice, Spinner, SelectControl } = wp.components;
  const { __ } = wp.i18n;
  const apiFetch = wp.apiFetch;

  function QueueTab({ connectionStatus, addNotice }) {
    const [jobs, setJobs] = useState([]);
    const [loading, setLoading] = useState(true);
    const [filter, setFilter] = useState("all");
    const [page, setPage] = useState(1);
    const [totalPages, setTotalPages] = useState(1);

    useEffect(() => {
      loadJobs();
    }, [filter, page]);

    const loadJobs = async () => {
      setLoading(true);
      try {
        const params = new URLSearchParams({
          page: page.toString(),
        });
        if (filter !== "all") {
          params.append("status", filter);
        }

        const result = await apiFetch({
          path: `/vidtoarticle/v1/admin/jobs?${params}`,
        });

        setJobs(result.jobs || []);
        setTotalPages(result.total_pages || 1);
      } catch (error) {
        addNotice("error", __("Failed to load jobs", "vidtoarticle-publisher"));
      }
      setLoading(false);
    };

    const handleRetry = async (jobId) => {
      try {
        await apiFetch({
          path: `/vidtoarticle/v1/admin/jobs/${jobId}/retry`,
          method: "POST",
        });

        addNotice(
          "success",
          __("Job queued for retry", "vidtoarticle-publisher")
        );
        loadJobs();
      } catch (error) {
        addNotice(
          "error",
          error.message || __("Failed to retry job", "vidtoarticle-publisher")
        );
      }
    };

    if (!connectionStatus.connected) {
      return el(
        Notice,
        { status: "warning", isDismissible: false },
        __("Please connect to VidToArticle.com first", "vidtoarticle-publisher")
      );
    }

    return el(
      Fragment,
      null,
      el(
        "div",
        { className: "queue-filters" },
        el(SelectControl, {
          label: __("Filter by Status", "vidtoarticle-publisher"),
          value: filter,
          options: [
            { value: "all", label: __("All Jobs", "vidtoarticle-publisher") },
            {
              value: "pending",
              label: __("Pending", "vidtoarticle-publisher"),
            },
            {
              value: "published",
              label: __("Published", "vidtoarticle-publisher"),
            },
            { value: "failed", label: __("Failed", "vidtoarticle-publisher") },
          ],
          onChange: (value) => {
            setFilter(value);
            setPage(1);
          },
        })
      ),
      loading
        ? el(Spinner)
        : jobs.length === 0
          ? el(
              Notice,
              { status: "info", isDismissible: false },
              __("No jobs found", "vidtoarticle-publisher")
            )
          : el(
              Fragment,
              null,
              el(
                "table",
                { className: "wp-list-table widefat fixed striped" },
                el(
                  "thead",
                  null,
                  el(
                    "tr",
                    null,
                    el("th", null, __("Video", "vidtoarticle-publisher")),
                    el("th", null, __("Source", "vidtoarticle-publisher")),
                    el("th", null, __("Status", "vidtoarticle-publisher")),
                    el("th", null, __("Created", "vidtoarticle-publisher")),
                    el("th", null, __("Actions", "vidtoarticle-publisher"))
                  )
                ),
                el(
                  "tbody",
                  null,
                  jobs.map((job) =>
                    el(
                      "tr",
                      { key: job.id },
                      el(
                        "td",
                        null,
                        el("strong", null, job.video_title || job.video_id),
                        el("br"),
                        el(
                          "small",
                          null,
                          el(
                            "a",
                            {
                              href: job.video_url,
                              target: "_blank",
                              rel: "noopener noreferrer",
                            },
                            __("View Video", "vidtoarticle-publisher")
                          )
                        )
                      ),
                      el("td", null, job.source_name || "-"),
                      el(
                        "td",
                        null,
                        el(
                          "span",
                          { className: `status-${job.status}` },
                          job.status
                        )
                      ),
                      el("td", null, new Date(job.created_at).toLocaleString()),
                      el(
                        "td",
                        null,
                        job.status.includes("failed")
                          ? el(
                              Button,
                              {
                                variant: "secondary",
                                isSmall: true,
                                onClick: () => handleRetry(job.id),
                              },
                              __("Retry", "vidtoarticle-publisher")
                            )
                          : job.post_id
                            ? el(
                                "a",
                                {
                                  href: `/wp-admin/post.php?post=${job.post_id}&action=edit`,
                                  className: "button button-small",
                                },
                                __("View Post", "vidtoarticle-publisher")
                              )
                            : "-"
                      )
                    )
                  )
                )
              ),
              totalPages > 1 &&
                el(
                  "div",
                  { className: "tablenav" },
                  el(
                    "div",
                    { className: "tablenav-pages" },
                    el(
                      Button,
                      {
                        variant: "secondary",
                        disabled: page === 1,
                        onClick: () => setPage(page - 1),
                      },
                      "‹"
                    ),
                    " ",
                    el("span", null, `${page} / ${totalPages}`),
                    " ",
                    el(
                      Button,
                      {
                        variant: "secondary",
                        disabled: page === totalPages,
                        onClick: () => setPage(page + 1),
                      },
                      "›"
                    )
                  )
                )
            )
    );
  }

  window.VidToArticleTabs = window.VidToArticleTabs || {};
  window.VidToArticleTabs.QueueTab = QueueTab;
})();
