(function () {
  "use strict";

  const { createElement: el, Fragment, useState } = wp.element;
  const { Button, Card, CardBody, Notice } = wp.components;
  const { __ } = wp.i18n;
  const apiFetch = wp.apiFetch;

  function ConnectionTab({ connectionStatus, refreshConnection, addNotice }) {
    const [connecting, setConnecting] = useState(false);
    const [disconnecting, setDisconnecting] = useState(false);

    const handleConnect = async () => {
      setConnecting(true);
      try {
        const result = await apiFetch({
          path: "/vidtoarticle/v1/admin/connect",
          method: "POST",
        });

        if (result.success && result.url) {
          const width = 600;
          const height = 700;
          const left = (screen.width - width) / 2;
          const top = (screen.height - height) / 2;

          const popup = window.open(
            result.url,
            "vidtoarticle_oauth",
            `width=${width},height=${height},left=${left},top=${top},toolbar=no,menubar=no,scrollbars=yes`
          );

          const handleMessage = (event) => {
            const allowedOrigins = [
              "https://api.vidtoarticle.com",
              "https://www.vidtoarticle.com",
              "http://localhost:3000",
              window.location.origin,
            ];

            if (!allowedOrigins.includes(event.origin)) {
              return;
            }

            if (event.data.type === "vidtoarticle_oauth_success") {
              window.removeEventListener("message", handleMessage);
              clearInterval(checkPopup);

              if (popup && !popup.closed) {
                popup.close();
              }

              setConnecting(false);
              refreshConnection();
              addNotice(
                "success",
                __("Connected successfully!", "vidtoarticle-publisher")
              );
            } else if (event.data.type === "vidtoarticle_oauth_error") {
              window.removeEventListener("message", handleMessage);
              clearInterval(checkPopup);

              if (popup && !popup.closed) {
                popup.close();
              }

              setConnecting(false);
              addNotice(
                "error",
                __("Connection failed: ", "vidtoarticle-publisher") +
                  (event.data.error || "Unknown error")
              );
            }
          };

          window.addEventListener("message", handleMessage);

          const checkPopup = setInterval(() => {
            if (popup.closed) {
              clearInterval(checkPopup);
              window.removeEventListener("message", handleMessage);
              setConnecting(false);
            }
          }, 500);
        }
      } catch (error) {
        addNotice(
          "error",
          __("Connection failed: ", "vidtoarticle-publisher") + error.message
        );
        setConnecting(false);
      }
    };

    const handleDisconnect = async () => {
      if (
        !confirm(
          __(
            "Are you sure you want to disconnect? This will stop all monitoring and publishing.",
            "vidtoarticle-publisher"
          )
        )
      ) {
        return;
      }

      setDisconnecting(true);
      try {
        await apiFetch({
          path: "/vidtoarticle/v1/admin/disconnect",
          method: "POST",
        });

        addNotice(
          "success",
          __("Disconnected successfully", "vidtoarticle-publisher")
        );
        await refreshConnection();
      } catch (error) {
        addNotice(
          "error",
          __("Disconnect failed: ", "vidtoarticle-publisher") + error.message
        );
      }
      setDisconnecting(false);
    };

    return el(
      Card,
      null,
      el(
        CardBody,
        null,
        connectionStatus.connected
          ? el(
              Fragment,
              null,
              el(
                Notice,
                { status: "success", isDismissible: false },
                __("✓ Connected to VidToArticle.com", "vidtoarticle-publisher")
              ),
              el(
                "div",
                { className: "connection-info" },
                el(
                  "h3",
                  null,
                  __("Connection Status", "vidtoarticle-publisher")
                ),
                el(
                  "table",
                  { className: "widefat" },
                  el(
                    "tbody",
                    null,
                    el(
                      "tr",
                      null,
                      el("th", null, __("Domain", "vidtoarticle-publisher")),
                      el("td", null, connectionStatus.domain || "-")
                    ),
                    el(
                      "tr",
                      null,
                      el(
                        "th",
                        null,
                        __("API Key ID", "vidtoarticle-publisher")
                      ),
                      el(
                        "td",
                        null,
                        el("code", null, connectionStatus.api_key_id || "-")
                      )
                    ),
                    el(
                      "tr",
                      null,
                      el(
                        "th",
                        null,
                        __("Article Quota", "vidtoarticle-publisher")
                      ),
                      el(
                        "td",
                        null,
                        connectionStatus.quota
                          ? `${connectionStatus.quota.used} / ${connectionStatus.quota.limit === -1 ? "∞" : connectionStatus.quota.limit} (${connectionStatus.quota.period})`
                          : "-"
                      )
                    ),
                    el(
                      "tr",
                      null,
                      el(
                        "th",
                        null,
                        __("Active Sources", "vidtoarticle-publisher")
                      ),
                      el("td", null, connectionStatus.sources_count || 0)
                    ),
                    el(
                      "tr",
                      null,
                      el(
                        "th",
                        null,
                        __("Last Activity", "vidtoarticle-publisher")
                      ),
                      el(
                        "td",
                        null,
                        connectionStatus.last_activity
                          ? new Date(
                              connectionStatus.last_activity
                            ).toLocaleString()
                          : __("Never", "vidtoarticle-publisher")
                      )
                    )
                  )
                )
              ),
              el(
                "div",
                {
                  className: "connection-actions",
                  style: { marginTop: "20px" },
                },
                el(
                  Button,
                  {
                    variant: "secondary",
                    onClick: refreshConnection,
                  },
                  __("Refresh Status", "vidtoarticle-publisher")
                ),
                " ",
                el(
                  Button,
                  {
                    variant: "tertiary",
                    isDestructive: true,
                    onClick: handleDisconnect,
                    isBusy: disconnecting,
                  },
                  __("Disconnect", "vidtoarticle-publisher")
                )
              )
            )
          : el(
              Fragment,
              null,
              el(
                Notice,
                { status: "warning", isDismissible: false },
                __(
                  "Not connected to VidToArticle.com",
                  "vidtoarticle-publisher"
                )
              ),
              el(
                "div",
                { className: "connection-setup" },
                el(
                  "h3",
                  null,
                  __("Connect Your WordPress Site", "vidtoarticle-publisher")
                ),
                el(
                  "p",
                  null,
                  __(
                    "Connect to VidToArticle.com to automatically generate and publish articles from YouTube videos.",
                    "vidtoarticle-publisher"
                  )
                ),
                el(
                  "ol",
                  null,
                  el(
                    "li",
                    null,
                    __(
                      "Click the Connect button below",
                      "vidtoarticle-publisher"
                    )
                  ),
                  el(
                    "li",
                    null,
                    __(
                      "Sign in to your VidToArticle.com account",
                      "vidtoarticle-publisher"
                    )
                  ),
                  el(
                    "li",
                    null,
                    __(
                      "Authorize this WordPress site",
                      "vidtoarticle-publisher"
                    )
                  ),
                  el(
                    "li",
                    null,
                    __(
                      "You'll be redirected back here automatically",
                      "vidtoarticle-publisher"
                    )
                  )
                ),
                el(
                  Button,
                  {
                    variant: "primary",
                    onClick: handleConnect,
                    isBusy: connecting,
                  },
                  __("Connect to VidToArticle", "vidtoarticle-publisher")
                )
              )
            )
      )
    );
  }

  window.VidToArticleTabs = window.VidToArticleTabs || {};
  window.VidToArticleTabs.ConnectionTab = ConnectionTab;
})();
