(function () {
  "use strict";

  const { createElement: el, useEffect, useState } = wp.element;
  const { Notice, Spinner } = wp.components;
  const { __ } = wp.i18n;
  const apiFetch = wp.apiFetch;

  function ActivityTab({ addNotice }) {
    const [activity, setActivity] = useState([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
      loadActivity();
    }, []);

    const loadActivity = async () => {
      setLoading(true);
      try {
        const result = await apiFetch({
          path: "/vidtoarticle/v1/admin/activity",
        });
        setActivity(result.activity || []);
      } catch (error) {
        addNotice(
          "error",
          __("Failed to load activity", "vidtoarticle-publisher")
        );
      }
      setLoading(false);
    };

    if (loading) {
      return el(Spinner);
    }

    if (activity.length === 0) {
      return el(
        Notice,
        { status: "info", isDismissible: false },
        __("No activity yet", "vidtoarticle-publisher")
      );
    }

    return el(
      "table",
      { className: "wp-list-table widefat fixed striped" },
      el(
        "thead",
        null,
        el(
          "tr",
          null,
          el("th", null, __("Time", "vidtoarticle-publisher")),
          el("th", null, __("Action", "vidtoarticle-publisher")),
          el("th", null, __("Message", "vidtoarticle-publisher"))
        )
      ),
      el(
        "tbody",
        null,
        activity.map((item, index) => {
          let contextDetails = null;
          if (item.context) {
            try {
              const ctx =
                typeof item.context === "string"
                  ? JSON.parse(item.context)
                  : item.context;
              contextDetails = Object.keys(ctx).length > 0 ? ctx : null;
            } catch (e) {
              // Ignore invalid JSON payloads
            }
          }

          return el(
            "tr",
            { key: index },
            el("td", null, new Date(item.created_at).toLocaleString()),
            el("td", null, el("code", null, item.action)),
            el(
              "td",
              null,
              el(
                "div",
                null,
                el("div", null, item.message),
                contextDetails &&
                  el(
                    "details",
                    { style: { marginTop: "8px", fontSize: "0.9em" } },
                    el(
                      "summary",
                      { style: { cursor: "pointer", color: "#666" } },
                      "Details"
                    ),
                    el(
                      "pre",
                      {
                        style: {
                          background: "#f5f5f5",
                          padding: "8px",
                          borderRadius: "4px",
                          fontSize: "0.85em",
                          overflow: "auto",
                          marginTop: "4px",
                        },
                      },
                      JSON.stringify(contextDetails, null, 2)
                    )
                  )
              )
            )
          );
        })
      )
    );
  }

  window.VidToArticleTabs = window.VidToArticleTabs || {};
  window.VidToArticleTabs.ActivityTab = ActivityTab;
})();
