/**
 * Admin Interface - React Application
 * Using WordPress Components for modern UI
 */

(function () {
  "use strict";
  // Handle OAuth callback - if this is a popup callback, notify parent and close
  const urlParams = new URLSearchParams(window.location.search);
  if (urlParams.get("oauth_callback") === "1") {
    // Notify parent window of success/error
    if (window.opener) {
      const status = urlParams.get("status");
      const error = urlParams.get("error");

      if (status === "success") {
        window.opener.postMessage(
          {
            type: "vidtoarticle_oauth_success",
          },
          window.location.origin
        );
      } else if (error) {
        window.opener.postMessage(
          {
            type: "vidtoarticle_oauth_error",
            error: error,
          },
          window.location.origin
        );
      }
    }

    // Show a simple message
    const status = urlParams.get("status");
    if (status === "success") {
      document.body.innerHTML =
        '<div style="padding: 40px; text-align: center; font-family: sans-serif;"><h2>✓ Connection Successful</h2><p>This window will close automatically...</p></div>';
    } else {
      const error = urlParams.get("error") || "Unknown error";
      document.body.innerHTML =
        '<div style="padding: 40px; text-align: center; font-family: sans-serif;"><h2>✗ Connection Failed</h2><p>' +
        error +
        "</p><p>This window will close automatically...</p></div>";
    }

    // Close popup after a short delay
    setTimeout(() => {
      window.close();
    }, 1500);
    return;
  }

  const { createElement: el, Fragment, useState, useEffect } = wp.element;
  const { TabPanel, Notice, Spinner } = wp.components;
  const { __ } = wp.i18n;
  const apiFetch = wp.apiFetch;

  const componentRegistry = (window.VidToArticleComponents =
    window.VidToArticleComponents || {});
  const tabsRegistry = (window.VidToArticleTabs =
    window.VidToArticleTabs || {});

  const MODULE_CHECK_MAX = 3;

  const requiredTabs = {
    connection: "ConnectionTab",
    sources: "SourcesTab",
    queue: "QueueTab",
    activity: "ActivityTab",
  };

  const getMissingModules = () => {
    const missing = [];
    if (!componentRegistry.StatsBar) {
      missing.push("StatsBar");
    }
    Object.values(requiredTabs).forEach((tabName) => {
      if (!tabsRegistry[tabName]) {
        missing.push(tabName);
      }
    });
    return missing;
  };

  const renderMissingModuleNotice = (name) =>
    el(
      Notice,
      { status: "error", isDismissible: false },
      __("Component failed to load.", "vidtoarticle-publisher") + ` (${name})`
    );

  // Configure apiFetch with nonce from localized script.
  if (window.vidToArticleAdmin && window.vidToArticleAdmin.nonce) {
    apiFetch.use(
      apiFetch.createNonceMiddleware(window.vidToArticleAdmin.nonce)
    );
  }

  /**
   * Main App Component
   */
  function App() {
    const [loading, setLoading] = useState(true);
    const [loadError, setLoadError] = useState(null);
    const [connectionStatus, setConnectionStatus] = useState(null);
    const [statistics, setStatistics] = useState(null);
    const [notices, setNotices] = useState([]);
    const [missingModules, setMissingModules] = useState(getMissingModules());
    const [moduleChecks, setModuleChecks] = useState(0);

    useEffect(() => {
      loadInitialData();
    }, []);

    useEffect(() => {
      if (missingModules.length === 0 || moduleChecks >= MODULE_CHECK_MAX) {
        return;
      }
      const timer = setTimeout(() => {
        setMissingModules(getMissingModules());
        setModuleChecks((prev) => prev + 1);
      }, 150);
      return () => clearTimeout(timer);
    }, [missingModules, moduleChecks]);

    const loadInitialData = async () => {
      try {
        const [connStatus, stats] = await Promise.all([
          apiFetch({ path: "/vidtoarticle/v1/admin/connection" }),
          apiFetch({ path: "/vidtoarticle/v1/admin/statistics" }),
        ]);

        setConnectionStatus(connStatus);
        setStatistics(stats.stats);
        setLoading(false);
        setLoadError(null);
      } catch (error) {
        console.error("VidToArticle API Error:", error);
        setLoadError(error.message || "Failed to connect to REST API");
        setLoading(false);
      }
    };

    const addNotice = (type, message) => {
      const id = Date.now();
      setNotices((prev) => [...prev, { id, type, message }]);
      setTimeout(() => {
        setNotices((prev) => prev.filter((n) => n.id !== id));
      }, 5000);
    };

    const refreshConnection = async () => {
      try {
        const connStatus = await apiFetch({
          path: "/vidtoarticle/v1/admin/connection",
        });
        setConnectionStatus(connStatus);
      } catch (error) {
        addNotice(
          "error",
          __("Failed to refresh connection status", "vidtoarticle-publisher")
        );
      }
    };

    if (missingModules.length > 0) {
      if (moduleChecks < MODULE_CHECK_MAX) {
        return el(
          "div",
          { className: "vidtoarticle-admin-loading" },
          el(Spinner),
          el(
            "p",
            null,
            __("Loading interface assets...", "vidtoarticle-publisher")
          )
        );
      }

      return el(
        "div",
        { className: "vidtoarticle-admin-loading" },
        el(
          Notice,
          { status: "error", isDismissible: false },
          el(
            "p",
            null,
            __(
              "Some admin components failed to initialize.",
              "vidtoarticle-publisher"
            )
          ),
          el(
            "ul",
            null,
            missingModules.map((name) =>
              el(
                "li",
                { key: name },
                `${name} (${__("chunk", "vidtoarticle-publisher")})`
              )
            )
          ),
          el(
            "p",
            null,
            __(
              "Please refresh the page to retry loading.",
              "vidtoarticle-publisher"
            )
          )
        )
      );
    }

    if (loading) {
      return el(
        "div",
        { className: "vidtoarticle-admin-loading" },
        el(Spinner),
        el("p", null, __("Loading...", "vidtoarticle-publisher"))
      );
    }

    const tabs = [
      {
        name: "connection",
        title: __("Connection", "vidtoarticle-publisher"),
        icon: "admin-links",
      },
      {
        name: "sources",
        title: __("Monitored Sources", "vidtoarticle-publisher"),
        icon: "playlist-video",
      },
      {
        name: "queue",
        title: __("Article Queue", "vidtoarticle-publisher"),
        icon: "list-view",
      },
      {
        name: "activity",
        title: __("Activity Log", "vidtoarticle-publisher"),
        icon: "admin-generic",
      },
    ];

    const StatsBarComponent = componentRegistry.StatsBar;

    const renderTab = (tabName, props) => {
      const componentName = requiredTabs[tabName];
      const TabComponent = componentName ? tabsRegistry[componentName] : null;
      if (!TabComponent) {
        return renderMissingModuleNotice(componentName || tabName);
      }
      return el(TabComponent, props);
    };

    return el(
      Fragment,
      null,
      el(
        "div",
        { className: "vidtoarticle-admin-header" },
        el(
          "h1",
          null,
          el("span", { className: "dashicons dashicons-video-alt3" }),
          __("VidToArticle Auto-Publisher", "vidtoarticle-publisher")
        ),
        statistics &&
          StatsBarComponent &&
          el(StatsBarComponent, { stats: statistics })
      ),

      loadError &&
        el(
          "div",
          { className: "vidtoarticle-admin-errors" },
          el(Notice, { status: "error", isDismissible: true }, loadError)
        ),

      el(
        "div",
        { className: "vidtoarticle-admin-notices" },
        notices.map((notice) =>
          el(
            Notice,
            {
              key: notice.id,
              status: notice.type,
              isDismissible: true,
              onRemove: () =>
                setNotices((prev) => prev.filter((n) => n.id !== notice.id)),
            },
            notice.message
          )
        )
      ),

      el(
        TabPanel,
        {
          className: "vidtoarticle-admin-tabs",
          activeClass: "is-active",
          tabs: tabs,
        },
        (tab) => {
          switch (tab.name) {
            case "connection":
              return renderTab("connection", {
                connectionStatus,
                refreshConnection,
                addNotice,
              });
            case "sources":
              return renderTab("sources", {
                connectionStatus,
                addNotice,
              });
            case "queue":
              return renderTab("queue", {
                connectionStatus,
                addNotice,
              });
            case "activity":
              return renderTab("activity", {
                addNotice,
              });
            default:
              return null;
          }
        }
      )
    );
  }

  // Render the app
  const root = document.getElementById("vidtoarticle-admin-root");
  if (root) {
    wp.element.render(el(App), root);
  }
})();
