# VidToArticle Auto Publisher - WordPress Plugin

Automatically publish AI-generated articles from YouTube playlists and channels via VidToArticle.com.

## Features

### Core Functionality
- ✅ **OAuth-style connection** to VidToArticle.com account
- ✅ **HMAC-signed API requests** for maximum security
- ✅ **Webhook push notifications** for real-time article delivery
- ✅ **Fallback polling** (every 12 hours) as safety net
- ✅ **Multiple source monitoring** (playlists and channels)
- ✅ **Customizable publishing** (category, author, style per source)
- ✅ **Backfill support** (0-10 recent videos on setup)
- ✅ **Draft-first workflow** for content review
- ✅ **Quota monitoring** with clear user feedback
- ✅ **Activity log** with manual retry for failures

### Security
- HMAC-SHA256 request signing with timestamp validation
- Domain verification on backend
- Encrypted secret key storage
- Replay attack prevention
- All inputs sanitized, all outputs escaped
- WordPress nonce verification
- Capability checks (`manage_options`)

## Installation

### Requirements
- WordPress 5.8 or higher
- PHP 7.4 or higher
- Active VidToArticle.com account
- HTTPS enabled (required for API communication)

### Steps
1. Upload the plugin folder to `/wp-content/plugins/`
2. Activate the plugin through the 'Plugins' menu in WordPress
3. Navigate to Settings → VidToArticle Publisher
4. Click "Connect to VidToArticle.com"
5. Authorize the connection in your VidToArticle.com account
6. Add your first monitored playlist or channel

## Database Schema

### `wp_vidtoarticle_sources`
Stores monitored YouTube playlists/channels with publishing settings.

| Field | Type | Description |
|-------|------|-------------|
| id | bigint | Primary key |
| source_type | varchar(20) | 'playlist' or 'channel' |
| source_id | varchar(100) | YouTube playlist/channel ID |
| source_name | varchar(255) | Cached name |
| article_style | varchar(50) | Article generation style |
| wp_category | bigint | WordPress category ID |
| wp_author | bigint | WordPress author ID |
| posts_per_day | tinyint | Publishing limit |
| backfill_count | tinyint | Number of recent videos processed on setup |
| last_checked_video_id | varchar(100) | Most recent video processed |
| last_published_at | datetime | Last successful publish |
| is_active | tinyint | Active status |

### `wp_vidtoarticle_jobs`
Tracks individual video-to-article jobs.

| Field | Type | Description |
|-------|------|-------------|
| id | bigint | Primary key |
| source_id | bigint | Foreign key to sources |
| backend_job_id | varchar(100) | VidToArticle.com job ID |
| video_id | varchar(100) | YouTube video ID |
| video_url | varchar(500) | Full video URL |
| video_title | varchar(500) | Video title |
| status | varchar(50) | Job status |
| failure_reason | text | Error message if failed |
| retry_count | tinyint | Number of retry attempts |
| post_id | bigint | WordPress post ID if published |
| processed_at | datetime | Completion timestamp |

**Status Values:**
- `pending` - Awaiting processing
- `processing` - Article generation in progress
- `completed` - Ready for publication
- `published` - WordPress post created
- `failed_generation` - AI generation failed
- `failed_quota` - User quota exceeded
- `failed_delivery` - Webhook delivery failed

### `wp_vidtoarticle_activity`
Activity log for debugging and user transparency.

| Field | Type | Description |
|-------|------|-------------|
| id | bigint | Primary key |
| job_id | bigint | Related job ID (nullable) |
| action | varchar(50) | Action type |
| message | text | Log message |
| context | longtext | JSON context data |
| created_at | datetime | Timestamp |

## API Integration

### Endpoint: Webhook Receiver
`POST /wp-json/vidtoarticle/v1/webhook`

Receives article delivery notifications from VidToArticle.com backend.

**Security:**
- Validates HMAC signature
- Checks timestamp (rejects if >5 minutes old)
- Verifies job hasn't been processed before

**Payload:**
```json
{
  "timestamp": "2025-01-07T12:00:00Z",
  "signature": "abc123...",
  "data": {
    "job_id": "job_xyz",
    "video": {
      "id": "abc123",
      "url": "https://youtube.com/watch?v=abc123",
      "title": "Video Title",
      "thumbnail_url": "https://i.ytimg.com/...",
      "published_at": "2025-01-07T10:00:00Z"
    },
    "article": {
      "title": "Generated Article Title",
      "content": "# Full Markdown...",
      "excerpt": "Brief summary...",
      "style": "news",
      "word_count": 1500,
      "tags": ["AI", "Technology"],
      "suggested_categories": ["Tech News"],
      "reading_time_minutes": 6
    }
  }
}
```

### Endpoint: OAuth Callback
`GET /wp-json/vidtoarticle/v1/oauth/callback`

Handles OAuth-style connection completion.

## Architecture

### Webhook Push (Primary)
1. User adds YouTube source in WordPress
2. WordPress registers webhook URL with backend
3. Backend monitors YouTube playlist/channel
4. When new video detected → Generates article
5. Backend sends webhook to WordPress
6. WordPress validates → Creates post → Confirms receipt

### Fallback Polling (Safety Net)
- Runs every 12 hours via WP-Cron
- Calls `/api/wordpress/missed-jobs` endpoint
- Processes any articles that weren't delivered
- Ensures 99%+ reliability even if webhooks fail

## Development

### File Structure
```
vidtoarticle-publisher/
├── vidtoarticle-publisher.php (Main plugin file)
├── includes/
│   ├── class-api-client.php (HMAC authentication)
│   ├── class-oauth-handler.php (Connection flow)
│   ├── class-webhook-listener.php (Receive webhooks)
│   ├── class-fallback-poller.php (WP-Cron backup)
│   ├── class-post-publisher.php (Create WP posts)
│   ├── class-source-manager.php (Source CRUD)
│   └── class-queue-manager.php (Job tracking)
├── admin/
│   ├── class-settings-page.php (Admin UI)
│   └── views/ (Admin templates)
└── assets/ (CSS/JS)
```

### Code Standards
- Follows WordPress PHP Coding Standards
- All strings translatable (i18n ready)
- Proper escaping and sanitization
- Prepared SQL statements
- Semantic versioning

### Testing
```bash
# Install WordPress test suite
bash bin/install-wp-tests.sh wordpress_test root '' localhost latest

# Run tests
phpunit
```

## Troubleshooting

### Connection Issues
**Problem:** "Failed to connect to VidToArticle.com"
- Ensure HTTPS is enabled on your site
- Check that your site URL is publicly accessible
- Verify firewall isn't blocking outbound connections to api.vidtoarticle.com

### Webhook Not Received
**Problem:** Articles generated but not appearing in WordPress
- Check if fallback polling is enabled (it should catch missed webhooks)
- Verify your hosting provider doesn't block incoming webhooks
- Check activity log for error messages

### Quota Exceeded
**Problem:** "Publishing paused - quota limit reached"
- Upgrade your VidToArticle.com plan
- Publishing will auto-resume when quota refreshes (for monthly plans)
- Check quota status in plugin dashboard

## Support

- Documentation: https://docs.vidtoarticle.com/wordpress
- Support: support@vidtoarticle.com
- GitHub: https://github.com/vidtoarticle/wordpress-plugin

## License

GPL v2 or later

## Changelog

### 1.0.0 - 2025-01-07
- Initial release
- OAuth-style connection flow
- HMAC authentication
- Webhook delivery with fallback polling
- Multi-source monitoring
- Draft-first workflow
- Activity logging
