<?php
/**
 * Plugin Name: VidToArticle Auto Publisher
 * Plugin URI: https://vidtoarticle.com/wordpress
 * Description: Automatically publish AI-generated articles from YouTube playlists and channels via VidToArticle.com
 * Version: 1.6.6
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Author: VidToArticle
 * Author URI: https://vidtoarticle.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: vidtoarticle-publisher
 * Domain Path: /languages
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' )) {
	exit;
}

// Define plugin constants.
define( 'VIDTOARTICLE_VERSION', '1.6.6' );
define( 'VIDTOARTICLE_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'VIDTOARTICLE_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'VIDTOARTICLE_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

// Load the autoloader.
require_once VIDTOARTICLE_PLUGIN_DIR . 'includes/class-loader.php';
VidToArticle_Loader::register();
VidToArticle_Loader::load_dependencies();

/**
 * Main Plugin Class
 */
class VidToArticle_Publisher {

	/**
	 * Single instance of the class
	 *
	 * @var VidToArticle_Publisher
	 */
	protected static $instance = null;

	/**
	 * API Client instance
	 *
	 * @var VidToArticle_API_Client
	 */
	public $api_client;

	/**
	 * Webhook Listener instance
	 *
	 * @var VidToArticle_Webhook_Listener
	 */
	public $webhook_listener;

	/**
	 * API Handler instance (for REST endpoints)
	 *
	 * @var VidToArticle_API_Handler
	 */
	public $api_handler;

	/**
	 * Admin instance
	 *
	 * @var VidToArticle_Admin
	 */
	public $admin;

	/**
	 * REST Controller instance
	 *
	 * @var VidToArticle_REST_Controller
	 */
	public $rest_controller;

	/**
	 * Get main instance
	 *
	 * @return VidToArticle_Publisher
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		// Activation and deactivation hooks.
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );

		// Initialize components.
		add_action( 'plugins_loaded', array( $this, 'init' ) );

		// Register REST API endpoints.
		add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
	}

	/**
	 * Plugin activation
	 */
	public function activate() {
		// Create database tables.
		$this->create_tables();

		// Schedule fallback polling cron.
		if ( ! wp_next_scheduled( 'vidtoarticle_fallback_poll' ) ) {
			wp_schedule_event( time(), 'twicedaily', 'vidtoarticle_fallback_poll' );
		}

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation
	 */
	public function deactivate() {
		// Clear scheduled cron.
		wp_clear_scheduled_hook( 'vidtoarticle_fallback_poll' );

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Initialize plugin components
	 */
	public function init() {
		// Initialize API client (dependency for other components).
		$this->api_client = new VidToArticle_API_Client();

		// Initialize webhook listener.
		$this->webhook_listener = new VidToArticle_Webhook_Listener();

		// Initialize fallback poller.
		$fallback_poller = new VidToArticle_Fallback_Poller();

		// Initialize API handler (always loaded for REST endpoints).
		$this->api_handler = new VidToArticle_API_Handler();

		// Initialize admin interface if needed.
		if ( is_admin() ) {
			require_once VIDTOARTICLE_PLUGIN_DIR . 'admin/class-admin.php';
			$this->admin = new VidToArticle_Admin();
		}

		// Load text domain for translations.
		load_plugin_textdomain( 'vidtoarticle-publisher', false, dirname( VIDTOARTICLE_PLUGIN_BASENAME ) . '/languages' );
	}

	/**
	 * Register REST API routes
	 */
	public function register_rest_routes() {
		// Debug logging.
		error_log( '[VidToArticle] register_rest_routes() called' );

		// Ensure components are initialized.
		if ( ! $this->api_handler ) {
			error_log( '[VidToArticle] Creating new API Handler' );
			$this->api_handler = new VidToArticle_API_Handler();
		}
		if ( ! $this->webhook_listener ) {
			error_log( '[VidToArticle] Creating new Webhook Listener' );
			$this->webhook_listener = new VidToArticle_Webhook_Listener();
		}

		// Load REST controller if not already loaded.
		if ( ! class_exists( 'VidToArticle_REST_Controller' ) ) {
			error_log( '[VidToArticle] Loading REST Controller class' );
			require_once VIDTOARTICLE_PLUGIN_DIR . 'includes/class-rest-controller.php';
		}

		// Initialize and register REST controller.
		// Note: We use api_handler instead of admin to ensure routes work outside admin context.
		error_log( '[VidToArticle] Initializing REST Controller' );
		$this->rest_controller = new VidToArticle_REST_Controller( $this->api_handler, $this->webhook_listener );
		error_log( '[VidToArticle] Calling register_routes()' );
		$this->rest_controller->register_routes();
		error_log( '[VidToArticle] Routes registered successfully' );
	}

	/**
	 * Create database tables
	 */
	private function create_tables() {
		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();

		// Table for monitored sources.
		$sql_sources = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}vidtoarticle_sources (
			id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			source_type varchar(20) NOT NULL,
			source_id varchar(100) NOT NULL,
			source_name varchar(255) DEFAULT NULL,
			article_style varchar(50) NOT NULL,
			wp_category bigint(20) UNSIGNED DEFAULT NULL,
			wp_author bigint(20) UNSIGNED DEFAULT NULL,
			posts_per_day tinyint(2) NOT NULL DEFAULT 1,
			backfill_count tinyint(2) NOT NULL DEFAULT 0,
			last_checked_video_id varchar(100) DEFAULT NULL,
			last_published_at datetime DEFAULT NULL,
			is_active tinyint(1) NOT NULL DEFAULT 1,
			created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY source_type (source_type),
			KEY is_active (is_active)
		) $charset_collate;";

		// Table for job queue.
		$sql_jobs = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}vidtoarticle_jobs (
			id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			source_id bigint(20) UNSIGNED NOT NULL,
			backend_job_id varchar(100) DEFAULT NULL,
			video_id varchar(100) NOT NULL,
			video_url varchar(500) NOT NULL,
			video_title varchar(500) DEFAULT NULL,
			status varchar(50) NOT NULL DEFAULT 'pending',
			failure_reason text DEFAULT NULL,
			retry_count tinyint(2) NOT NULL DEFAULT 0,
			post_id bigint(20) UNSIGNED DEFAULT NULL,
			created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			processed_at datetime DEFAULT NULL,
			PRIMARY KEY (id),
			KEY source_id (source_id),
			KEY status (status),
			KEY video_id (video_id),
			KEY backend_job_id (backend_job_id)
		) $charset_collate;";

		// Table for activity log.
		$sql_activity = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}vidtoarticle_activity (
			id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			job_id bigint(20) UNSIGNED DEFAULT NULL,
			action varchar(50) NOT NULL,
			message text NOT NULL,
			context longtext DEFAULT NULL,
			created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY job_id (job_id),
			KEY action (action),
			KEY created_at (created_at)
		) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql_sources );
		dbDelta( $sql_jobs );
		dbDelta( $sql_activity );

		// Store database version.
		update_option( 'vidtoarticle_db_version', '1.0.0' );
	}
}

/**
 * Get main instance of the plugin
 *
 * @return VidToArticle_Publisher
 */
function vidtoarticle_publisher() {
	return VidToArticle_Publisher::instance();
}

// Initialize the plugin.
vidtoarticle_publisher();
