<?php
/**
 * REST API Controller
 *
 * Centralized REST API route registration and handling.
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * REST Controller Class
 *
 * Handles all REST API endpoints for the plugin.
 */
class VidToArticle_REST_Controller {

	/**
	 * Namespace for REST routes
	 *
	 * @var string
	 */
	private $namespace = 'vidtoarticle/v1';

	/**
	 * API Handler instance
	 *
	 * @var VidToArticle_API_Handler
	 */
	private $api_handler;

	/**
	 * Webhook Listener instance
	 *
	 * @var VidToArticle_Webhook_Listener
	 */
	private $webhook_listener;

	/**
	 * Constructor
	 *
	 * @param VidToArticle_API_Handler      $api_handler      API handler instance.
	 * @param VidToArticle_Webhook_Listener $webhook_listener Webhook listener instance.
	 */
	public function __construct( $api_handler = null, $webhook_listener = null ) {
		$this->api_handler      = $api_handler;
		$this->webhook_listener = $webhook_listener;
	}

	/**
	 * Register all REST routes
	 */
	public function register_routes() {
		error_log( '[VidToArticle] REST Controller register_routes() called' );
		error_log( '[VidToArticle] Namespace: ' . $this->namespace );
		error_log( '[VidToArticle] API Handler present: ' . ( $this->api_handler ? 'YES' : 'NO' ) );

		// Webhook endpoint.
		register_rest_route(
			$this->namespace,
			'/webhook',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'handle_webhook' ),
				'permission_callback' => '__return_true', // HMAC validation done in handler.
			)
		);
		error_log( '[VidToArticle] Webhook route registered' );

		// Register admin routes if API handler is available.
		if ( $this->api_handler ) {
			error_log( '[VidToArticle] Registering admin routes...' );
			$this->register_admin_routes();
			error_log( '[VidToArticle] Admin routes registered' );
		} else {
			error_log( '[VidToArticle] WARNING: API Handler not available, admin routes NOT registered' );
		}
	}

	/**
	 * Register admin-specific REST routes
	 */
	private function register_admin_routes() {
		// Connection endpoints.
		register_rest_route(
			$this->namespace,
			'/admin/connection',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->api_handler, 'get_connection_status' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/connect',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this->api_handler, 'initiate_connection' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/disconnect',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this->api_handler, 'disconnect' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		// Source endpoints.
		register_rest_route(
			$this->namespace,
			'/admin/sources',
			array(
				array(
					'methods'             => 'GET',
					'callback'            => array( $this->api_handler, 'get_sources' ),
					'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
				),
				array(
					'methods'             => 'POST',
					'callback'            => array( $this->api_handler, 'create_source' ),
					'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/sources/(?P<id>[a-zA-Z0-9]+)',
			array(
				array(
					'methods'             => 'PUT',
					'callback'            => array( $this->api_handler, 'update_source' ),
					'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
				),
				array(
					'methods'             => 'DELETE',
					'callback'            => array( $this->api_handler, 'delete_source' ),
					'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
				),
			)
		);

		// Jobs endpoints.
		register_rest_route(
			$this->namespace,
			'/admin/jobs',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->api_handler, 'get_jobs' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/jobs/(?P<id>\d+)/retry',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this->api_handler, 'retry_job' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/jobs/(?P<id>\d+)/cancel',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this->api_handler, 'cancel_job' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		// Statistics endpoint.
		register_rest_route(
			$this->namespace,
			'/admin/statistics',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->api_handler, 'get_statistics' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		// Activity log endpoint.
		register_rest_route(
			$this->namespace,
			'/admin/activity',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->api_handler, 'get_activity_log' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);
	}

	/**
	 * Handle webhook callback
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response or error.
	 */
	public function handle_webhook( $request ) {
		if ( $this->webhook_listener ) {
			return $this->webhook_listener->handle_webhook( $request );
		}

		return new WP_Error(
			'webhook_unavailable',
			__( 'Webhook listener not initialized', 'vidtoarticle-publisher' ),
			array( 'status' => 500 )
		);
	}
}
