<?php
/**
 * API Handler
 *
 * Handles all REST API endpoint logic.
 * This class is always loaded (not just in admin context) so REST routes work properly.
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * API Handler Class
 *
 * Separated from admin UI to ensure REST routes always work.
 */
class VidToArticle_API_Handler {

	/**
	 * OAuth Handler instance
	 *
	 * @var VidToArticle_OAuth_Handler
	 */
	private $oauth_handler;

	/**
	 * Source Manager instance
	 *
	 * @var VidToArticle_Source_Manager
	 */
	private $source_manager;

	/**
	 * Queue Manager instance
	 *
	 * @var VidToArticle_Queue_Manager
	 */
	private $queue_manager;

	/**
	 * API Client instance
	 *
	 * @var VidToArticle_API_Client
	 */
	private $api_client;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->oauth_handler  = new VidToArticle_OAuth_Handler();
		$this->source_manager = new VidToArticle_Source_Manager();
		$this->queue_manager  = new VidToArticle_Queue_Manager();
		$this->api_client     = new VidToArticle_API_Client();
	}

	/**
	 * Check admin permission
	 */
	public function check_admin_permission() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * Get connection status
	 */
	public function get_connection_status() {
		$is_connected = $this->api_client->is_connected();

		$response = array(
			'connected' => $is_connected,
		);

		if ( $is_connected ) {
			$status = $this->api_client->get_connection_status();

			if ( ! is_wp_error( $status ) ) {
				$response = array_merge( $response, $status );
			}
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Initiate connection
	 */
	public function initiate_connection() {
		$url = VidToArticle_OAuth_Handler::get_connection_url();

		return rest_ensure_response(
			array(
				'success' => true,
				'url'     => $url,
			)
		);
	}

	/**
	 * Disconnect
	 */
	public function disconnect() {
		// Try to revoke connection via API if we have a secret key.
		if ( $this->api_client->is_connected() ) {
			$response = $this->api_client->revoke_connection();
			// If revoke fails, log but don't stop disconnect - still clear local data.
			if ( is_wp_error( $response ) ) {
				error_log( 'VidToArticle: Failed to revoke API key: ' . $response->get_error_message() );
			}
		}

		// Clear local secret key and connection status.
		delete_option( 'vidtoarticle_secret_key' );
		delete_option( 'vidtoarticle_connection_status' );

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Disconnected successfully', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Get sources
	 */
	public function get_sources() {
		// Check if plugin is connected before making API calls.
		if ( ! $this->api_client->is_connected() ) {
			// Return empty sources array if not authenticated.
			return rest_ensure_response(
				array(
					'success' => true,
					'sources' => array(),
					'count' => 0,
					'synced_with_backend' => false,
					'not_authenticated' => true,
				)
			);
		}

		$remote_response = $this->api_client->get_sources();

		if ( is_wp_error( $remote_response ) ) {
			$sources = $this->source_manager->get_sources();

			return rest_ensure_response(
				array(
					'success' => true,
					'sources' => $sources,
					'synced_with_backend' => false,
					'error' => $remote_response->get_error_message(),
				)
			);
		}

		$backend_sources = isset( $remote_response['sources'] ) && is_array( $remote_response['sources'] )
			? $remote_response['sources']
			: array();
		$normalized_sources = array_map( array( $this, 'transform_backend_source' ), $backend_sources );

		return rest_ensure_response(
			array(
				'success' => true,
				'sources' => $normalized_sources,
				'count' => isset( $remote_response['count'] ) ? intval( $remote_response['count'] ) : count( $normalized_sources ),
				'synced_with_backend' => true,
			)
		);
	}

	/**
	 * Create source
	 */
	public function create_source( $request ) {
		$params = $request->get_json_params();

		// Validate required parameters.
		if ( empty( $params['source_type'] ) || empty( $params['source_url'] ) || empty( $params['article_style'] ) ) {
			return new WP_Error(
				'missing_parameters',
				__( 'Missing required parameters', 'vidtoarticle-publisher' ),
				array( 'status' => 400 )
			);
		}

		// Extract YouTube ID from URL
		$source_id = $this->extract_youtube_id( $params['source_url'], $params['source_type'] );
		if ( is_wp_error( $source_id ) ) {
			return $source_id;
		}

		$result = $this->source_manager->create_source(
			array(
				'source_type'    => $params['source_type'],
				'source_id'      => $source_id,
				'article_style'  => $params['article_style'],
				'posts_per_day'  => isset( $params['posts_per_day'] ) ? $params['posts_per_day'] : 1,
				'backfill_count' => isset( $params['backfill_count'] ) ? $params['backfill_count'] : 0,
				'wp_category'    => isset( $params['wp_category'] ) ? $params['wp_category'] : null,
				'wp_author'      => isset( $params['wp_author'] ) ? $params['wp_author'] : null,
			)
		);

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'create_failed',
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		return rest_ensure_response(
			array(
				'success'   => true,
				'source_id' => $result,
				'message'   => __( 'Source created successfully', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Update source
	 */
	public function update_source( $request ) {
		$id     = $request->get_param( 'id' );
		$params = $request->get_json_params();

		$result = $this->source_manager->update_source( $id, $params );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'update_failed',
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Source updated successfully', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Delete source
	 */
	public function delete_source( $request ) {
		$id = $request->get_param( 'id' );

		$result = $this->source_manager->delete_source( $id );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'delete_failed',
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Source deleted successfully', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Get jobs
	 */
	public function get_jobs( $request ) {
		$status = $request->get_param( 'status' );
		$page   = $request->get_param( 'page' ) ? intval( $request->get_param( 'page' ) ) : 1;
		$limit  = 20;

		$args = array(
			'status' => $status,
			'limit'  => $limit,
			'offset' => ( $page - 1 ) * $limit,
		);

		$jobs  = $this->queue_manager->get_jobs( $args );
		$total = $this->queue_manager->get_job_count( $status );

		return rest_ensure_response(
			array(
				'success'      => true,
				'jobs'         => $jobs,
				'total'        => $total,
				'page'         => $page,
				'total_pages'  => ceil( $total / $limit ),
			)
		);
	}

	/**
	 * Retry job
	 */
	public function retry_job( $request ) {
		$id = $request->get_param( 'id' );

		$result = $this->queue_manager->retry_job( $id );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'retry_failed',
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Job queued for retry', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Cancel job
	 */
	public function cancel_job( $request ) {
		$id = $request->get_param( 'id' );

		$result = $this->queue_manager->cancel_job( $id );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'cancel_failed',
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Job cancelled successfully', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Get activity log
	 */
	public function get_activity_log( $request ) {
		$page  = $request->get_param( 'page' ) ? intval( $request->get_param( 'page' ) ) : 1;
		$limit = 50;

		$args = array(
			'limit'  => $limit,
			'offset' => ( $page - 1 ) * $limit,
		);

		$activity = $this->queue_manager->get_activity( $args );

		return rest_ensure_response(
			array(
				'success'  => true,
				'activity' => $activity,
			)
		);
	}

	/**
	 * Get statistics
	 */
	public function get_statistics() {
		$stats = $this->queue_manager->get_statistics();

		return rest_ensure_response(
			array(
				'success' => true,
				'stats'   => $stats,
			)
		);
	}

	/**
	 * Extract YouTube ID from URL
	 *
	 * @param string $url YouTube URL.
	 * @param string $type Source type (playlist or channel).
	 * @return string|WP_Error YouTube ID or error.
	 */
	private function extract_youtube_id( $url, $type ) {
		// Playlist patterns
		if ( 'playlist' === $type ) {
			// https://www.youtube.com/playlist?list=PLxxxxxx
			// https://youtube.com/playlist?list=PLxxxxxx
			if ( preg_match( '/[?&]list=([a-zA-Z0-9_-]+)/', $url, $matches ) ) {
				return $matches[1];
			}
		}

		// Channel patterns
		if ( 'channel' === $type ) {
			// https://www.youtube.com/channel/UCxxxxxx
			// https://youtube.com/channel/UCxxxxxx
			if ( preg_match( '/youtube\.com\/channel\/([a-zA-Z0-9_-]+)/', $url, $matches ) ) {
				return $matches[1];
			}

			// https://www.youtube.com/@username
			// https://youtube.com/@username
			if ( preg_match( '/youtube\.com\/@([a-zA-Z0-9_-]+)/', $url, $matches ) ) {
				return '@' . $matches[1];
			}

			// https://www.youtube.com/c/customname
			// https://youtube.com/c/customname
			if ( preg_match( '/youtube\.com\/c\/([a-zA-Z0-9_-]+)/', $url, $matches ) ) {
				return $matches[1];
			}

			// https://www.youtube.com/user/username
			// https://youtube.com/user/username
			if ( preg_match( '/youtube\.com\/user\/([a-zA-Z0-9_-]+)/', $url, $matches ) ) {
				return $matches[1];
			}
		}

		return new WP_Error(
			'invalid_url',
			sprintf(
				/* translators: %s: source type */
				__( 'Could not extract YouTube %s ID from URL. Please check the URL format.', 'vidtoarticle-publisher' ),
				$type
			)
		);
	}

	/**
	 * Normalize backend source payload for WordPress UI consumption.
	 *
	 * @param array $source Backend source payload.
	 * @return array
	 */
	private function transform_backend_source( $source ) {
		$settings = isset( $source['settings'] ) && is_array( $source['settings'] ) ? $source['settings'] : array();
		$playlist_url = null;
		$channel_url  = null;

		if ( isset( $source['sourceType'], $source['sourceId'] ) ) {
			if ( 'playlist' === $source['sourceType'] ) {
				$playlist_url = $this->build_playlist_url( $source['sourceId'] );
			} elseif ( 'channel' === $source['sourceType'] ) {
				$channel_url = $this->build_channel_url( $source['sourceId'] );
			}
		}

		return array(
			'id' => isset( $source['id'] ) ? $source['id'] : '',
			'source_id' => isset( $source['sourceId'] ) ? $source['sourceId'] : '',
			'source_type' => isset( $source['sourceType'] ) ? $source['sourceType'] : '',
			'source_name' => isset( $source['sourceName'] ) ? $source['sourceName'] : '',
			'article_style' => isset( $source['articleStyle'] ) ? $source['articleStyle'] : '',
			'posts_per_day' => isset( $source['postsPerDay'] ) ? intval( $source['postsPerDay'] ) : null,
			'backfill_count' => isset( $source['backfillCount'] ) ? intval( $source['backfillCount'] ) : null,
			'is_active' => ! empty( $source['isActive'] ) ? 1 : 0,
			'wp_category' => isset( $settings['wpCategory'] ) ? $settings['wpCategory'] : null,
			'wp_author' => isset( $settings['wpAuthor'] ) ? $settings['wpAuthor'] : null,
			'last_checked_video_id' => isset( $source['lastCheckedVideoId'] ) ? $source['lastCheckedVideoId'] : null,
			'created_at' => isset( $source['createdAt'] ) ? $source['createdAt'] : null,
			'updated_at' => isset( $source['updatedAt'] ) ? $source['updatedAt'] : null,
			'domain' => isset( $source['domain'] ) ? $source['domain'] : null,
			'user_id' => isset( $source['userId'] ) ? $source['userId'] : null,
			'playlist_url' => $playlist_url,
			'channel_url' => $channel_url,
		);
	}

	/**
	 * Build playlist URL from playlist ID.
	 *
	 * @param string $playlist_id Playlist ID.
	 * @return string
	 */
	private function build_playlist_url( $playlist_id ) {
		return sprintf( 'https://www.youtube.com/playlist?list=%s', rawurlencode( $playlist_id ) );
	}

	/**
	 * Build channel URL from channel identifier.
	 *
	 * @param string $channel_id Channel ID or handle.
	 * @return string
	 */
	private function build_channel_url( $channel_id ) {
		if ( 0 === strpos( $channel_id, '@' ) ) {
			return 'https://www.youtube.com/' . $channel_id;
		}

		return sprintf( 'https://www.youtube.com/channel/%s', rawurlencode( $channel_id ) );
	}
}

