(function () {
  "use strict";

  const { createElement: el, Fragment, useEffect, useState } = wp.element;
  const {
    Button,
    Notice,
    Spinner,
    SelectControl,
    TextControl,
    Modal,
    __experimentalNumberControl: NumberControl,
  } = wp.components;
  const { __ } = wp.i18n;
  const apiFetch = wp.apiFetch;

  function AddSourceModal({ onClose, onSuccess, addNotice }) {
    const [sourceType, setSourceType] = useState("playlist");
    const [sourceUrl, setSourceUrl] = useState("");
    const [articleStyle, setArticleStyle] = useState("news");
    const [postsPerDay, setPostsPerDay] = useState(1);
    const [backfillCount, setBackfillCount] = useState(0);
    const [wpCategory, setWpCategory] = useState("");
    const [wpAuthor, setWpAuthor] = useState("");
    const [saving, setSaving] = useState(false);

    const handleSave = async () => {
      if (!sourceUrl) {
        addNotice(
          "error",
          __("Please enter a YouTube URL", "vidtoarticle-publisher")
        );
        return;
      }

      setSaving(true);
      try {
        const response = await apiFetch({
          path: "/vidtoarticle/v1/admin/sources",
          method: "POST",
          data: {
            source_type: sourceType,
            source_url: sourceUrl,
            article_style: articleStyle,
            posts_per_day: parseInt(postsPerDay, 10),
            backfill_count: parseInt(backfillCount, 10),
            wp_category: wpCategory || null,
            wp_author: wpAuthor || null,
          },
        });

        addNotice(
          "success",
          response.message ||
            __("Source added successfully", "vidtoarticle-publisher")
        );
        onSuccess();
      } catch (error) {
        addNotice(
          "error",
          error.message || __("Failed to add source", "vidtoarticle-publisher")
        );
        setSaving(false);
      }
    };

    const styles = window.vidToArticleAdmin.styles || {};
    const styleOptions = Object.keys(styles).map((key) => ({
      value: key,
      label: styles[key],
    }));

    return el(
      Modal,
      {
        title: __("Add New Source", "vidtoarticle-publisher"),
        onRequestClose: onClose,
        className: "vidtoarticle-modal",
      },
      el(
        "div",
        { className: "vidtoarticle-modal-content" },
        el(SelectControl, {
          label: __("Source Type", "vidtoarticle-publisher"),
          value: sourceType,
          options: [
            {
              value: "playlist",
              label: __("YouTube Playlist", "vidtoarticle-publisher"),
            },
            {
              value: "channel",
              label: __("YouTube Channel", "vidtoarticle-publisher"),
            },
          ],
          onChange: setSourceType,
        }),
        el(TextControl, {
          label: __("YouTube URL", "vidtoarticle-publisher"),
          value: sourceUrl,
          onChange: setSourceUrl,
          placeholder:
            sourceType === "playlist"
              ? "https://www.youtube.com/playlist?list=..."
              : "https://www.youtube.com/@channelname",
          help:
            sourceType === "playlist"
              ? __(
                  "Paste the full URL of the YouTube playlist",
                  "vidtoarticle-publisher"
                )
              : __(
                  "Paste the channel URL (e.g., youtube.com/@channelname)",
                  "vidtoarticle-publisher"
                ),
        }),
        el(SelectControl, {
          label: __("Article Style", "vidtoarticle-publisher"),
          value: articleStyle,
          options: styleOptions,
          onChange: setArticleStyle,
        }),
        el(NumberControl, {
          label: __("Posts Per Day", "vidtoarticle-publisher"),
          value: postsPerDay,
          onChange: setPostsPerDay,
          min: 1,
          max: 10,
          help: __(
            "Maximum number of articles to publish per day from this source",
            "vidtoarticle-publisher"
          ),
        }),
        el(NumberControl, {
          label: __("Backfill Count", "vidtoarticle-publisher"),
          value: backfillCount,
          onChange: setBackfillCount,
          min: 0,
          max: 10,
          help: __(
            "Number of recent videos to process immediately (0-10)",
            "vidtoarticle-publisher"
          ),
        }),
        el(SelectControl, {
          label: __("WordPress Category", "vidtoarticle-publisher"),
          value: wpCategory,
          options: [
            {
              value: "",
              label: __("-- Select Category --", "vidtoarticle-publisher"),
            },
            ...(window.vidToArticleAdmin.categories || []),
          ],
          onChange: setWpCategory,
        }),
        el(SelectControl, {
          label: __("WordPress Author", "vidtoarticle-publisher"),
          value: wpAuthor,
          options: [
            {
              value: "",
              label: __("-- Select Author --", "vidtoarticle-publisher"),
            },
            ...(window.vidToArticleAdmin.authors || []),
          ],
          onChange: setWpAuthor,
        }),
        el(
          "div",
          { className: "modal-actions" },
          el(
            Button,
            {
              variant: "secondary",
              onClick: onClose,
            },
            __("Cancel", "vidtoarticle-publisher")
          ),
          " ",
          el(
            Button,
            {
              variant: "primary",
              onClick: handleSave,
              isBusy: saving,
            },
            __("Add Source", "vidtoarticle-publisher")
          )
        )
      )
    );
  }

  function SourcesTab({ connectionStatus, addNotice }) {
    const [sources, setSources] = useState([]);
    const [loading, setLoading] = useState(true);
    const [showAddModal, setShowAddModal] = useState(false);

    useEffect(() => {
      loadSources();
    }, []);

    const loadSources = async () => {
      setLoading(true);
      try {
        const result = await apiFetch({
          path: "/vidtoarticle/v1/admin/sources",
        });
        setSources(result.sources || []);
      } catch (error) {
        addNotice(
          "error",
          __("Failed to load sources: ", "vidtoarticle-publisher") +
            (error.message || "Unknown error")
        );
      }
      setLoading(false);
    };

    const handleDeleteSource = async (id) => {
      if (
        !confirm(
          __(
            "Are you sure you want to delete this source?",
            "vidtoarticle-publisher"
          )
        )
      ) {
        return;
      }

      try {
        await apiFetch({
          path: `/vidtoarticle/v1/admin/sources/${id}`,
          method: "DELETE",
        });

        addNotice(
          "success",
          __("Source deleted successfully", "vidtoarticle-publisher")
        );
        loadSources();
      } catch (error) {
        addNotice(
          "error",
          __("Failed to delete source", "vidtoarticle-publisher")
        );
      }
    };

    if (!connectionStatus.connected) {
      return el(
        Notice,
        { status: "warning", isDismissible: false },
        __("Please connect to VidToArticle.com first", "vidtoarticle-publisher")
      );
    }

    return el(
      Fragment,
      null,
      el(
        "div",
        { className: "sources-header" },
        el(
          Button,
          {
            variant: "primary",
            onClick: () => setShowAddModal(true),
          },
          __("Add New Source", "vidtoarticle-publisher")
        )
      ),
      loading
        ? el(Spinner)
        : sources.length === 0
          ? el(
              Notice,
              { status: "info", isDismissible: false },
              __(
                'No sources configured yet. Click "Add New Source" to get started.',
                "vidtoarticle-publisher"
              )
            )
          : el(
              "table",
              { className: "wp-list-table widefat fixed striped" },
              el(
                "thead",
                null,
                el(
                  "tr",
                  null,
                  el("th", null, __("Source", "vidtoarticle-publisher")),
                  el("th", null, __("Type", "vidtoarticle-publisher")),
                  el("th", null, __("Style", "vidtoarticle-publisher")),
                  el("th", null, __("Posts/Day", "vidtoarticle-publisher")),
                  el("th", null, __("Status", "vidtoarticle-publisher")),
                  el("th", null, __("Actions", "vidtoarticle-publisher"))
                )
              ),
              el(
                "tbody",
                null,
                sources.map((source) =>
                  el(
                    "tr",
                    { key: source.id },
                    el(
                      "td",
                      null,
                      el(
                        "strong",
                        null,
                        source.source_name || source.source_id
                      ),
                      el("br"),
                      el(
                        "small",
                        null,
                        source.source_id,
                        source.source_type === "playlist" &&
                          source.playlist_url &&
                          el(
                            Fragment,
                            null,
                            " · ",
                            el(
                              "a",
                              {
                                href: source.playlist_url,
                                target: "_blank",
                                rel: "noopener noreferrer",
                              },
                              __("Open playlist", "vidtoarticle-publisher")
                            )
                          ),
                        source.source_type === "channel" &&
                          source.channel_url &&
                          el(
                            Fragment,
                            null,
                            " · ",
                            el(
                              "a",
                              {
                                href: source.channel_url,
                                target: "_blank",
                                rel: "noopener noreferrer",
                              },
                              __("Open channel", "vidtoarticle-publisher")
                            )
                          )
                      )
                    ),
                    el("td", null, source.source_type),
                    el("td", null, source.article_style),
                    el("td", null, source.posts_per_day),
                    el(
                      "td",
                      null,
                      source.is_active == 1
                        ? el("span", { className: "status-active" }, "● Active")
                        : el(
                            "span",
                            { className: "status-inactive" },
                            "○ Inactive"
                          )
                    ),
                    el(
                      "td",
                      null,
                      el(
                        Button,
                        {
                          variant: "tertiary",
                          isDestructive: true,
                          isSmall: true,
                          onClick: () => handleDeleteSource(source.id),
                        },
                        __("Delete", "vidtoarticle-publisher")
                      )
                    )
                  )
                )
              )
            ),
      showAddModal &&
        el(AddSourceModal, {
          onClose: () => setShowAddModal(false),
          onSuccess: () => {
            setShowAddModal(false);
            loadSources();
            addNotice(
              "success",
              __("Source added successfully", "vidtoarticle-publisher")
            );
          },
          addNotice,
        })
    );
  }

  window.VidToArticleTabs = window.VidToArticleTabs || {};
  window.VidToArticleTabs.SourcesTab = SourcesTab;
})();
