<?php
/**
 * Admin Interface
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin Class
 */
class VidToArticle_Admin {

	/**
	 * OAuth Handler instance
	 *
	 * @var VidToArticle_OAuth_Handler
	 */
	private $oauth_handler;

	/**
	 * Source Manager instance
	 *
	 * @var VidToArticle_Source_Manager
	 */
	private $source_manager;

	/**
	 * Queue Manager instance
	 *
	 * @var VidToArticle_Queue_Manager
	 */
	private $queue_manager;

	/**
	 * API Client instance
	 *
	 * @var VidToArticle_API_Client
	 */
	private $api_client;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->oauth_handler  = new VidToArticle_OAuth_Handler();
		$this->source_manager = new VidToArticle_Source_Manager();
		$this->queue_manager  = new VidToArticle_Queue_Manager();
		$this->api_client     = new VidToArticle_API_Client();

		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		// Note: REST routes are now registered via VidToArticle_REST_Controller.
	}

	/**
	 * Add admin menu
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'VidToArticle', 'vidtoarticle-publisher' ),
			__( 'VidToArticle', 'vidtoarticle-publisher' ),
			'manage_options',
			'vidtoarticle-publisher',
			array( $this, 'render_admin_page' ),
			plugin_dir_url( __FILE__ ) . 'assets/vidtoarticle-icon.svg',
			30
		);
	}

	/**
	 * Enqueue admin assets
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		// Only load on our admin page.
		if ( 'toplevel_page_vidtoarticle-publisher' !== $hook ) {
			return;
		}

		// Enqueue WordPress components (React, etc).
		$shared_script_deps = array( 'wp-element', 'wp-components', 'wp-api-fetch', 'wp-i18n', 'wp-data', 'wp-notices' );
		foreach ( $shared_script_deps as $dep ) {
			wp_enqueue_script( $dep );
		}

		// Enqueue WordPress styles.
		wp_enqueue_style( 'wp-components' );

		$module_scripts = array(
			'vidtoarticle-admin-component-statsbar' => 'components/StatsBar.js',
			'vidtoarticle-admin-tab-connection'    => 'tabs/connection-tab.js',
			'vidtoarticle-admin-tab-sources'       => 'tabs/sources-tab.js',
			'vidtoarticle-admin-tab-queue'         => 'tabs/queue-tab.js',
			'vidtoarticle-admin-tab-activity'      => 'tabs/activity-tab.js',
		);

		$module_handles = array();
		foreach ( $module_scripts as $handle => $relative_path ) {
			wp_enqueue_script(
				$handle,
				plugin_dir_url( __FILE__ ) . 'js/' . $relative_path,
				$shared_script_deps,
				VIDTOARTICLE_VERSION,
				true
			);
			$module_handles[] = $handle;
		}

		$admin_script_deps = array_merge( $shared_script_deps, $module_handles );

		// Enqueue our custom admin script.
		wp_enqueue_script(
			'vidtoarticle-admin',
			plugin_dir_url( __FILE__ ) . 'js/admin.js',
			$admin_script_deps,
			VIDTOARTICLE_VERSION,
			true
		);

		// Enqueue custom admin styles.
		wp_enqueue_style(
			'vidtoarticle-admin',
			plugin_dir_url( __FILE__ ) . 'css/admin.css',
			array( 'wp-components' ),
			VIDTOARTICLE_VERSION
		);

		// Localize script with data.
		wp_localize_script(
			'vidtoarticle-admin',
			'vidToArticleAdmin',
			array(
				'restUrl'      => rest_url( 'vidtoarticle/v1' ),
				'nonce'        => wp_create_nonce( 'wp_rest' ),
				'isConnected'  => $this->api_client->is_connected(),
				'styles'       => array(
					'news'             => __( 'News Style', 'vidtoarticle-publisher' ),
					'bombastic'        => __( 'Bombastic Style', 'vidtoarticle-publisher' ),
					'tabloid'          => __( 'Tabloid Style', 'vidtoarticle-publisher' ),
					'technical'        => __( 'Technical Style', 'vidtoarticle-publisher' ),
					'academic'         => __( 'Academic Style', 'vidtoarticle-publisher' ),
					'casual'           => __( 'Casual Style', 'vidtoarticle-publisher' ),
					'listicle'         => __( 'Listicle Style', 'vidtoarticle-publisher' ),
				),
				'categories'   => $this->get_categories_list(),
				'authors'      => $this->get_authors_list(),
			)
		);
	}

	/**
	 * Render admin page
	 */
	public function render_admin_page() {
		// Check if REST API is accessible.
		$api_check = $this->check_rest_api();

		?>
		<div class="wrap">
			<h1>
				<span class="dashicons dashicons-video-alt3" style="font-size: 32px;"></span>
				<?php esc_html_e( 'VidToArticle Auto-Publisher', 'vidtoarticle-publisher' ); ?>
			</h1>

			<?php if ( ! $api_check['success'] ) : ?>
				<div class="notice notice-error">
					<h2><?php esc_html_e( 'REST API Connection Error', 'vidtoarticle-publisher' ); ?></h2>
					<p><strong><?php echo esc_html( $api_check['error'] ); ?></strong></p>
					<p><?php esc_html_e( 'The plugin interface requires WordPress REST API to function. Please check:', 'vidtoarticle-publisher' ); ?></p>
					<ol>
						<li><?php esc_html_e( 'Permalink Settings: Go to Settings → Permalinks and ensure it\'s not set to "Plain"', 'vidtoarticle-publisher' ); ?></li>
						<li><?php esc_html_e( 'Click "Save Changes" on Permalinks page to flush rewrite rules', 'vidtoarticle-publisher' ); ?></li>
						<li><?php esc_html_e( 'Check if REST API works: Visit /wp-json/ on your site', 'vidtoarticle-publisher' ); ?></li>
						<li><?php esc_html_e( 'Disable conflicting plugins (security plugins that block REST API)', 'vidtoarticle-publisher' ); ?></li>
					</ol>
					<p>
						<a href="<?php echo esc_url( admin_url( 'options-permalink.php' ) ); ?>" class="button button-primary">
							<?php esc_html_e( 'Fix Permalinks', 'vidtoarticle-publisher' ); ?>
						</a>
						<a href="<?php echo esc_url( rest_url() ); ?>" class="button" target="_blank">
							<?php esc_html_e( 'Test REST API', 'vidtoarticle-publisher' ); ?>
						</a>
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=vidtoarticle-publisher' ) ); ?>" class="button">
							<?php esc_html_e( 'Retry', 'vidtoarticle-publisher' ); ?>
						</a>
					</p>
					<details style="margin-top: 20px;">
						<summary style="cursor: pointer; font-weight: bold;">
							<?php esc_html_e( 'Technical Details', 'vidtoarticle-publisher' ); ?>
						</summary>
						<pre style="background: #f5f5f5; padding: 15px; margin-top: 10px; overflow-x: auto;"><?php echo esc_html( print_r( $api_check, true ) ); ?></pre>
					</details>
				</div>
			<?php endif; ?>

			<div id="vidtoarticle-admin-root"></div>

			<?php if ( ! $api_check['success'] ) : ?>
				<noscript>
					<div class="notice notice-warning">
						<p><?php esc_html_e( 'JavaScript is required for this plugin to function properly.', 'vidtoarticle-publisher' ); ?></p>
					</div>
				</noscript>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Check if REST API is accessible
	 *
	 * @return array Status and error message.
	 */
	private function check_rest_api() {
		// Check if REST API is enabled.
		if ( ! function_exists( 'rest_url' ) ) {
			return array(
				'success' => false,
				'error'   => 'WordPress REST API is not available on this installation.',
			);
		}

		// Use internal REST API request to bypass cookie/nonce issues during admin check.
		// This simulates what the JavaScript will do, but server-side.
		$request = new WP_REST_Request( 'GET', '/vidtoarticle/v1/admin/connection' );

		// Set current user context for the request.
		$request->set_param( 'context', 'edit' );

		// Get the REST server.
		$server = rest_get_server();

		// Dispatch the request internally (no HTTP call needed).
		$response = $server->dispatch( $request );

		// Check if it's an error.
		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'error'   => 'REST API error: ' . $response->get_error_message(),
				'details' => array(
					'endpoint' => '/vidtoarticle/v1/admin/connection',
					'error'    => $response->get_error_message(),
				),
			);
		}

		// Get status code from response.
		$status_code = $response->get_status();

		// 401/403 is okay - means routes are registered (auth will work from browser with proper cookies).
		// 200 is okay - means everything works.
		// 404 is bad - means routes aren't registered.
		if ( ! in_array( $status_code, array( 200, 401, 403 ), true ) ) {
			return array(
				'success' => false,
				'error'   => sprintf( 'REST API returned status code: %d', $status_code ),
				'details' => array(
					'endpoint'    => '/vidtoarticle/v1/admin/connection',
					'status_code' => $status_code,
					'body'        => $response->get_data(),
				),
			);
		}

		// If we get here, routes are registered. The JavaScript can handle auth.
		return array( 'success' => true );
	}

	/**
	 * Register REST API routes for admin
	 */
	public function register_rest_routes() {
		// Connection endpoints.
		register_rest_route(
			'vidtoarticle/v1',
			'/admin/connection',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_connection_status' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			'vidtoarticle/v1',
			'/admin/connect',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'initiate_connection' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			'vidtoarticle/v1',
			'/admin/disconnect',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'disconnect' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		// Source endpoints.
		register_rest_route(
			'vidtoarticle/v1',
			'/admin/sources',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_sources' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			'vidtoarticle/v1',
			'/admin/sources',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'create_source' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			'vidtoarticle/v1',
			'/admin/sources/(?P<id>\d+)',
			array(
				'methods'             => 'PUT',
				'callback'            => array( $this, 'update_source' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			'vidtoarticle/v1',
			'/admin/sources/(?P<id>\d+)',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'delete_source' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		// Jobs endpoints.
		register_rest_route(
			'vidtoarticle/v1',
			'/admin/jobs',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_jobs' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			'vidtoarticle/v1',
			'/admin/jobs/(?P<id>\d+)/retry',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'retry_job' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		// Activity endpoints.
		register_rest_route(
			'vidtoarticle/v1',
			'/admin/activity',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_activity' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		// Statistics endpoint.
		register_rest_route(
			'vidtoarticle/v1',
			'/admin/statistics',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_statistics' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);
	}

	/**
	 * Check admin permission
	 */
	public function check_admin_permission() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * Get connection status
	 */
	public function get_connection_status() {
		$is_connected = $this->oauth_handler->is_connected();

		$response = array(
			'connected' => $is_connected,
		);

		if ( $is_connected ) {
			$status = $this->api_client->get_connection_status();

			if ( ! is_wp_error( $status ) ) {
				$response = array_merge( $response, $status );
			}
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Initiate connection
	 */
	public function initiate_connection() {
		$url = VidToArticle_OAuth_Handler::get_connection_url();

		return rest_ensure_response(
			array(
				'success' => true,
				'url'     => $url,
			)
		);
	}

	/**
	 * Disconnect
	 */
	public function disconnect() {
		$result = $this->oauth_handler->disconnect();

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'disconnect_failed',
				$result->get_error_message(),
				array( 'status' => 500 )
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Disconnected successfully', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Get sources
	 */
	public function get_sources() {
		$sources = $this->source_manager->list_sources();

		return rest_ensure_response(
			array(
				'success' => true,
				'sources' => $sources,
			)
		);
	}

	/**
	 * Create source
	 */
	public function create_source( $request ) {
		$params = $request->get_json_params();

		$result = $this->source_manager->create_source(
			$params['source_type'],
			$params['source_url'],
			$params['article_style'],
			$params['posts_per_day'],
			$params['backfill_count'],
			array(
				'wp_category' => isset( $params['wp_category'] ) ? $params['wp_category'] : null,
				'wp_author'   => isset( $params['wp_author'] ) ? $params['wp_author'] : null,
			)
		);

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'create_failed',
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		return rest_ensure_response(
			array(
				'success'   => true,
				'source_id' => $result,
				'message'   => __( 'Source created successfully', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Update source
	 */
	public function update_source( $request ) {
		$id     = $request->get_param( 'id' );
		$params = $request->get_json_params();

		$result = $this->source_manager->update_source( $id, $params );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'update_failed',
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Source updated successfully', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Delete source
	 */
	public function delete_source( $request ) {
		$id = $request->get_param( 'id' );

		$result = $this->source_manager->delete_source( $id );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'delete_failed',
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Source deleted successfully', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Get jobs
	 */
	public function get_jobs( $request ) {
		$status = $request->get_param( 'status' );
		$page   = $request->get_param( 'page' ) ? intval( $request->get_param( 'page' ) ) : 1;
		$limit  = 20;

		$args = array(
			'status' => $status,
			'limit'  => $limit,
			'offset' => ( $page - 1 ) * $limit,
		);

		$jobs  = $this->queue_manager->get_jobs( $args );
		$total = $this->queue_manager->get_job_count( $status );

		return rest_ensure_response(
			array(
				'success'    => true,
				'jobs'       => $jobs,
				'total'      => $total,
				'page'       => $page,
				'total_pages' => ceil( $total / $limit ),
			)
		);
	}

	/**
	 * Retry job
	 */
	public function retry_job( $request ) {
		$id = $request->get_param( 'id' );

		$result = $this->queue_manager->retry_job( $id );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'retry_failed',
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Job queued for retry', 'vidtoarticle-publisher' ),
			)
		);
	}

	/**
	 * Get activity
	 */
	public function get_activity( $request ) {
		$page  = $request->get_param( 'page' ) ? intval( $request->get_param( 'page' ) ) : 1;
		$limit = 50;

		$args = array(
			'limit'  => $limit,
			'offset' => ( $page - 1 ) * $limit,
		);

		$activity = $this->queue_manager->get_activity( $args );

		return rest_ensure_response(
			array(
				'success'  => true,
				'activity' => $activity,
			)
		);
	}

	/**
	 * Get statistics
	 */
	public function get_statistics() {
		$stats = $this->queue_manager->get_statistics();

		return rest_ensure_response(
			array(
				'success' => true,
				'stats'   => $stats,
			)
		);
	}

	/**
	 * Get categories list
	 */
	private function get_categories_list() {
		$categories = get_categories(
			array(
				'hide_empty' => false,
			)
		);

		$list = array();
		foreach ( $categories as $category ) {
			$list[] = array(
				'value' => $category->term_id,
				'label' => $category->name,
			);
		}

		return $list;
	}

	/**
	 * Get authors list
	 */
	private function get_authors_list() {
		$authors = get_users(
			array(
				'who' => 'authors',
			)
		);

		$list = array();
		foreach ( $authors as $author ) {
			$list[] = array(
				'value' => $author->ID,
				'label' => $author->display_name,
			);
		}

		return $list;
	}
}
