<?php
/**
 * Uninstall VidToArticle Auto Publisher
 *
 * This file is executed when the plugin is uninstalled via WordPress admin.
 * It removes all plugin data including database tables, options, transients,
 * scheduled events, and post meta.
 *
 * @package VidToArticle_Publisher
 * @since 1.6.2
 */

// Exit if uninstall not called from WordPress.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

// Security: Verify WordPress environment.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main uninstall function
 *
 * Removes all plugin data from the WordPress installation.
 */
function vidtoarticle_uninstall() {
	global $wpdb;

	// Require capabilities check (extra security layer).
	if ( ! current_user_can( 'activate_plugins' ) ) {
		return;
	}

	// Log the uninstallation (before we delete the activity table).
	vidtoarticle_log_uninstall_start();

	// 1. Remove all scheduled cron jobs.
	vidtoarticle_clear_cron_jobs();

	// 2. Delete all plugin options.
	vidtoarticle_delete_options();

	// 3. Delete all plugin transients.
	vidtoarticle_delete_transients();

	// 4. Delete all post meta created by the plugin.
	vidtoarticle_delete_post_meta();

	// 5. Delete all custom database tables.
	vidtoarticle_drop_tables();

	// Clear any cached data.
	wp_cache_flush();
}

/**
 * Log uninstallation start (for debugging purposes)
 */
function vidtoarticle_log_uninstall_start() {
	global $wpdb;

	// Only log if WP_DEBUG_LOG is enabled.
	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log( '[VidToArticle] Plugin uninstallation started' );
	}

	// Log to database activity table (if it exists).
	$table_name = $wpdb->prefix . 'vidtoarticle_activity';
	if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ) === $table_name ) {
		$wpdb->insert(
			$table_name,
			array(
				'action'     => 'plugin_uninstall',
				'message'    => 'Plugin uninstallation initiated',
				'context'    => wp_json_encode(
					array(
						'user_id' => get_current_user_id(),
						'time'    => current_time( 'mysql' ),
					)
				),
				'created_at' => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%s', '%s' )
		);
	}
}

/**
 * Clear all scheduled cron jobs
 */
function vidtoarticle_clear_cron_jobs() {
	// Clear the fallback polling cron job.
	$timestamp = wp_next_scheduled( 'vidtoarticle_fallback_poll' );
	if ( $timestamp ) {
		wp_unschedule_event( $timestamp, 'vidtoarticle_fallback_poll' );
	}

	// Clear all instances of the cron hook (in case of duplicates).
	wp_clear_scheduled_hook( 'vidtoarticle_fallback_poll' );

	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log( '[VidToArticle] Cleared scheduled cron jobs' );
	}
}

/**
 * Delete all plugin options
 */
function vidtoarticle_delete_options() {
	// List of all plugin options.
	$options = array(
		'vidtoarticle_db_version',
		'vidtoarticle_secret_key',
		'vidtoarticle_default_post_status',
		'vidtoarticle_connection_status',
	);

	// Delete each option.
	foreach ( $options as $option ) {
		delete_option( $option );
	}

	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log( '[VidToArticle] Deleted ' . count( $options ) . ' options' );
	}
}

/**
 * Delete all plugin transients
 *
 * Removes both standard transients and site transients.
 */
function vidtoarticle_delete_transients() {
	global $wpdb;

	// Standard transients to delete.
	$transients = array(
		'vidtoarticle_oauth_token',
		'vidtoarticle_oauth_state',
		'vidtoarticle_admin_notice',
	);

	// Delete standard transients.
	foreach ( $transients as $transient ) {
		delete_transient( $transient );
		delete_site_transient( $transient );
	}

	// Delete all idempotency transients (dynamic keys).
	// These are stored with pattern: vidtoarticle_idempotency_*
	$wpdb->query(
		$wpdb->prepare(
			"DELETE FROM {$wpdb->options}
			WHERE option_name LIKE %s
			OR option_name LIKE %s",
			$wpdb->esc_like( '_transient_vidtoarticle_idempotency_' ) . '%',
			$wpdb->esc_like( '_transient_timeout_vidtoarticle_idempotency_' ) . '%'
		)
	);

	// For multisite: delete site transients.
	if ( is_multisite() ) {
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->sitemeta}
				WHERE meta_key LIKE %s
				OR meta_key LIKE %s",
				$wpdb->esc_like( '_site_transient_vidtoarticle_idempotency_' ) . '%',
				$wpdb->esc_like( '_site_transient_timeout_vidtoarticle_idempotency_' ) . '%'
			)
		);
	}

	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log( '[VidToArticle] Deleted transients' );
	}
}

/**
 * Delete all post meta created by the plugin
 *
 * Removes meta from posts and attachments.
 */
function vidtoarticle_delete_post_meta() {
	global $wpdb;

	// List of all plugin post meta keys.
	$post_meta_keys = array(
		// Post meta.
		'_vidtoarticle_video_url',
		'_vidtoarticle_video_id',
		'_vidtoarticle_style',
		'_vidtoarticle_word_count',
		'_vidtoarticle_reading_time',
		'_vidtoarticle_source_url',
		'_vidtoarticle_video_id_backend',
		'_vidtoarticle_generated_at_backend',
		'_vidtoarticle_generator',
		'_vidtoarticle_featured_image_source',
		'_vidtoarticle_featured_image_set_at',
		// Attachment meta.
		'_vidtoarticle_image_source',
		'_vidtoarticle_photographer',
		'_vidtoarticle_photographer_url',
		'_vidtoarticle_original_url',
		'_vidtoarticle_download_url',
	);

	// Delete each meta key from all posts.
	foreach ( $post_meta_keys as $meta_key ) {
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->postmeta} WHERE meta_key = %s",
				$meta_key
			)
		);
	}

	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log( '[VidToArticle] Deleted ' . count( $post_meta_keys ) . ' types of post meta' );
	}
}

/**
 * Drop all custom database tables
 *
 * Removes tables created by the plugin.
 */
function vidtoarticle_drop_tables() {
	global $wpdb;

	// Get table names.
	$tables = array(
		$wpdb->prefix . 'vidtoarticle_sources',
		$wpdb->prefix . 'vidtoarticle_jobs',
		$wpdb->prefix . 'vidtoarticle_activity',
	);

	// Drop each table.
	foreach ( $tables as $table ) {
		$wpdb->query( "DROP TABLE IF EXISTS {$table}" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log( '[VidToArticle] Dropped ' . count( $tables ) . ' database tables' );
		error_log( '[VidToArticle] Plugin uninstallation completed successfully' );
	}
}

// Execute the uninstall function.
vidtoarticle_uninstall();
