<?php
/**
 * REST API Controller
 *
 * Centralized REST API route registration and handling.
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * REST Controller Class
 *
 * Handles all REST API endpoints for the plugin.
 */
class VidToArticle_REST_Controller {

	/**
	 * Namespace for REST routes
	 *
	 * @var string
	 */
	private $namespace = 'vidtoarticle/v1';

	/**
	 * API Handler instance
	 *
	 * @var VidToArticle_API_Handler
	 */
	private $api_handler;

	/**
	 * Webhook Listener instance
	 *
	 * @var VidToArticle_Webhook_Listener
	 */
	private $webhook_listener;

	/**
	 * Constructor
	 *
	 * @param VidToArticle_API_Handler      $api_handler      API handler instance.
	 * @param VidToArticle_Webhook_Listener $webhook_listener Webhook listener instance.
	 */
	public function __construct( $api_handler = null, $webhook_listener = null ) {
		$this->api_handler      = $api_handler;
		$this->webhook_listener = $webhook_listener;
	}

	/**
	 * Register all REST routes
	 */
	public function register_routes() {
		error_log( '[VidToArticle] REST Controller register_routes() called' );
		error_log( '[VidToArticle] Namespace: ' . $this->namespace );
		error_log( '[VidToArticle] API Handler present: ' . ( $this->api_handler ? 'YES' : 'NO' ) );

		// Webhook endpoint.
		register_rest_route(
			$this->namespace,
			'/webhook',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'handle_webhook' ),
				'permission_callback' => '__return_true', // HMAC validation done in handler.
			)
		);
		error_log( '[VidToArticle] Webhook route registered' );

		// Register admin routes if API handler is available.
		if ( $this->api_handler ) {
			error_log( '[VidToArticle] Registering admin routes...' );
			$this->register_admin_routes();
			error_log( '[VidToArticle] Admin routes registered' );
		} else {
			error_log( '[VidToArticle] WARNING: API Handler not available, admin routes NOT registered' );
		}

		// Health check endpoint (HMAC authenticated).
		register_rest_route(
			$this->namespace,
			'/health',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'health_check' ),
				'permission_callback' => array( $this, 'verify_hmac_permission' ),
			)
		);

		// Remote disconnect endpoint (HMAC authenticated).
		// Called by backend when user disconnects from VidToArticle.com dashboard.
		register_rest_route(
			$this->namespace,
			'/disconnect',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'remote_disconnect' ),
				'permission_callback' => array( $this, 'verify_hmac_permission' ),
			)
		);

		// Register public/HMAC routes (Authors & Categories).
		register_rest_route(
			$this->namespace,
			'/categories',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_categories' ),
				'permission_callback' => array( $this, 'verify_hmac_permission' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/authors',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_authors' ),
				'permission_callback' => array( $this, 'verify_hmac_permission' ),
			)
		);
	}


	/**
	 * Register admin-specific REST routes
	 */
	private function register_admin_routes() {
		// Connection endpoints.
		register_rest_route(
			$this->namespace,
			'/admin/connection',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->api_handler, 'get_connection_status' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/connect',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this->api_handler, 'initiate_connection' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/disconnect',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this->api_handler, 'disconnect' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		// Source endpoints.
		register_rest_route(
			$this->namespace,
			'/admin/sources',
			array(
				array(
					'methods'             => 'GET',
					'callback'            => array( $this->api_handler, 'get_sources' ),
					'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
				),
				array(
					'methods'             => 'POST',
					'callback'            => array( $this->api_handler, 'create_source' ),
					'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/sources/(?P<id>[a-zA-Z0-9]+)',
			array(
				array(
					'methods'             => 'PUT',
					'callback'            => array( $this->api_handler, 'update_source' ),
					'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
				),
				array(
					'methods'             => 'DELETE',
					'callback'            => array( $this->api_handler, 'delete_source' ),
					'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
				),
			)
		);

		// Jobs endpoints.
		register_rest_route(
			$this->namespace,
			'/admin/jobs',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->api_handler, 'get_jobs' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/jobs/(?P<id>\d+)/retry',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this->api_handler, 'retry_job' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/jobs/(?P<id>\d+)/cancel',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this->api_handler, 'cancel_job' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		// Statistics endpoint.
		register_rest_route(
			$this->namespace,
			'/admin/statistics',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->api_handler, 'get_statistics' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		// Activity log endpoint.
		register_rest_route(
			$this->namespace,
			'/admin/activity',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->api_handler, 'get_activity_log' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);

		// Backend logs endpoint (logs from VidToArticle.com server).
		register_rest_route(
			$this->namespace,
			'/admin/backend-logs',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this->api_handler, 'get_backend_logs' ),
				'permission_callback' => array( $this->api_handler, 'check_admin_permission' ),
			)
		);
	}

	/**
	 * Handle webhook callback
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response or error.
	 */
	public function handle_webhook( $request ) {
		if ( $this->webhook_listener ) {
			return $this->webhook_listener->handle_webhook( $request );
		}

		return new WP_Error(
			'webhook_unavailable',
			__( 'Webhook listener not initialized', 'vidtoarticle-publisher' ),
			array( 'status' => 500 )
		);
	}

	/**
	 * Get categories list
	 *
	 * @return WP_REST_Response
	 */
	public function get_categories() {
		$categories = get_categories(
			array(
				'hide_empty' => false,
			)
		);

		$list = array();
		foreach ( $categories as $category ) {
			$list[] = array(
				'id'    => $category->term_id,
				'name'  => $category->name,
				'slug'  => $category->slug,
				'count' => $category->count,
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'data'    => $list,
			)
		);
	}

	/**
	 * Get authors list
	 *
	 * @return WP_REST_Response
	 */
	public function get_authors() {
		$authors = get_users(
			array(
				'who' => 'authors',
			)
		);

		$list = array();
		foreach ( $authors as $author ) {
			$list[] = array(
				'id'   => $author->ID,
				'name' => $author->display_name,
				'slug' => $author->user_nicename,
			);
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'data'    => $list,
			)
		);
	}

	/**
	 * Health check endpoint
	 *
	 * Returns plugin status and version for connection verification.
	 *
	 * @return WP_REST_Response
	 */
	public function health_check() {
		// Load plugin.php for get_plugin_data() - not available on front-end requests.
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$plugin_data = get_plugin_data( VIDTOARTICLE_PLUGIN_FILE );
		$is_connected = ! empty( get_option( 'vidtoarticle_secret_key' ) );

		return rest_ensure_response(
			array(
				'success'       => true,
				'status'        => 'healthy',
				'connected'     => $is_connected,
				'plugin_version' => $plugin_data['Version'] ?? VIDTOARTICLE_VERSION,
				'wordpress_version' => get_bloginfo( 'version' ),
				'site_name'     => get_bloginfo( 'name' ),
				'timestamp'     => time(),
			)
		);
	}

	/**
	 * Remote disconnect endpoint
	 *
	 * Called by VidToArticle.com backend when user disconnects from dashboard.
	 * Clears local credentials so the plugin shows disconnected state.
	 *
	 * @return WP_REST_Response
	 */
	public function remote_disconnect() {
		// Clear the secret key and connection status (same as local disconnect)
		delete_option( 'vidtoarticle_secret_key' );
		delete_option( 'vidtoarticle_connection_status' );

		// Log the remote disconnect
		error_log( '[VidToArticle] Plugin disconnected remotely from VidToArticle.com dashboard' );

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => 'Disconnected successfully',
			)
		);
	}

	/**
	 * Verify HMAC signature for general requests
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return true|WP_Error True if valid, WP_Error otherwise.
	 */

	public function verify_hmac_permission( $request ) {
		// Get headers.
		$timestamp = $request->get_header( 'X-VidToArticle-Timestamp' );
		$signature = $request->get_header( 'X-VidToArticle-Signature' );

		if ( empty( $timestamp ) || empty( $signature ) ) {
			return new WP_Error(
				'missing_headers',
				__( 'Missing authentication headers', 'vidtoarticle-publisher' ),
				array( 'status' => 401 )
			);
		}

		// Check timestamp (prevent replay attacks).
		$current_time = time();
		$max_age      = 300; // 5 minutes
		$age          = abs( $current_time - intval( $timestamp ) );

		if ( $age > $max_age ) {
			return new WP_Error(
				'expired_timestamp',
				__( 'Timestamp too old', 'vidtoarticle-publisher' ),
				array( 'status' => 401 )
			);
		}

		// Get secret key.
		$secret_key = get_option( 'vidtoarticle_secret_key' );

		if ( empty( $secret_key ) ) {
			return new WP_Error(
				'not_connected',
				__( 'Plugin not connected', 'vidtoarticle-publisher' ),
				array( 'status' => 401 )
			);
		}

		// Calculate expected signature.
		$body              = $request->get_body();
		$body_hash         = hash( 'sha256', $body );
		$method            = $request->get_method();
		// Use full path with /wp-json prefix to match how backend generates signatures
		// $request->get_route() returns route without /wp-json prefix (e.g., /vidtoarticle/v1/health)
		// but backend signs with full path (e.g., /wp-json/vidtoarticle/v1/health)
		$route             = '/wp-json' . $request->get_route();
		
		$signature_string  = $timestamp . $method . $route . $body_hash;
		$expected_signature = hash_hmac( 'sha256', $signature_string, $secret_key );

		// Compare signatures (timing-safe).
		if ( ! hash_equals( $expected_signature, $signature ) ) {
			return new WP_Error(
				'invalid_signature',
				__( 'Invalid signature', 'vidtoarticle-publisher' ),
				array( 'status' => 401 )
			);
		}

		return true;
	}
}
