<?php
/**
 * Queue Manager
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Queue Manager Class
 */
class VidToArticle_Queue_Manager {

	/**
	 * Get jobs
	 *
	 * @param array $args Query arguments.
	 * @return array Jobs list.
	 */
	public function get_jobs( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'status'   => null,
			'limit'    => 20,
			'offset'   => 0,
			'order_by' => 'created_at',
			'order'    => 'DESC',
		);

		$args = wp_parse_args( $args, $defaults );

		$where = array( '1=1' );
		$prepare_args = array();

		if ( ! empty( $args['status'] ) ) {
			$where[] = 'status = %s';
			$prepare_args[] = $args['status'];
		}

		if ( ! empty( $args['source_id'] ) ) {
			$where[] = 'source_id = %d';
			$prepare_args[] = intval( $args['source_id'] );
		}

		$where_clause = implode( ' AND ', $where );

		$query = "SELECT j.*, s.source_name, s.article_style
				FROM {$wpdb->prefix}vidtoarticle_jobs j
				LEFT JOIN {$wpdb->prefix}vidtoarticle_sources s ON j.source_id = s.id
				WHERE {$where_clause}
				ORDER BY {$args['order_by']} {$args['order']}
				LIMIT %d OFFSET %d";

		$prepare_args[] = intval( $args['limit'] );
		$prepare_args[] = intval( $args['offset'] );

		if ( ! empty( $prepare_args ) && count( $prepare_args ) > 2 ) {
			$prepared = $wpdb->prepare( $query, $prepare_args );
		} else {
			$prepared = $wpdb->prepare( $query, intval( $args['limit'] ), intval( $args['offset'] ) );
		}

		return $wpdb->get_results( $prepared );
	}

	/**
	 * Get job count by status
	 *
	 * @param string $status Job status.
	 * @return int Count.
	 */
	public function get_job_count( $status = null ) {
		global $wpdb;

		if ( $status ) {
			return $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->prefix}vidtoarticle_jobs WHERE status = %s",
					$status
				)
			);
		}

		return $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->prefix}vidtoarticle_jobs"
		);
	}

	/**
	 * Get activity log
	 *
	 * @param array $args Query arguments.
	 * @return array Activity entries.
	 */
	public function get_activity( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'action' => null,
			'limit'  => 50,
			'offset' => 0,
		);

		$args = wp_parse_args( $args, $defaults );

		$where = array( '1=1' );
		$prepare_args = array();

		if ( ! empty( $args['action'] ) ) {
			$where[] = 'action = %s';
			$prepare_args[] = $args['action'];
		}

		$where_clause = implode( ' AND ', $where );

		$query = "SELECT * FROM {$wpdb->prefix}vidtoarticle_activity
				WHERE {$where_clause}
				ORDER BY created_at DESC
				LIMIT %d OFFSET %d";

		$prepare_args[] = intval( $args['limit'] );
		$prepare_args[] = intval( $args['offset'] );

		if ( count( $prepare_args ) > 2 ) {
			$prepared = $wpdb->prepare( $query, $prepare_args );
		} else {
			$prepared = $wpdb->prepare( $query, intval( $args['limit'] ), intval( $args['offset'] ) );
		}

		return $wpdb->get_results( $prepared );
	}

	/**
	 * Get statistics
	 *
	 * @return array Statistics data.
	 */
	public function get_statistics() {
		global $wpdb;

		$stats = array();

		// Total posts published.
		$stats['total_published'] = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->prefix}vidtoarticle_jobs WHERE status = 'published'"
		);

		// Pending jobs.
		$stats['pending'] = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->prefix}vidtoarticle_jobs WHERE status = 'pending'"
		);

		// Failed jobs.
		$stats['failed'] = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->prefix}vidtoarticle_jobs
			WHERE status LIKE 'failed%'"
		);

		// Active sources.
		$stats['active_sources'] = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->prefix}vidtoarticle_sources WHERE is_active = 1"
		);

		// Last 7 days activity.
		$stats['last_7_days'] = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$wpdb->prefix}vidtoarticle_jobs
				WHERE status = 'published' AND processed_at >= %s",
				date( 'Y-m-d H:i:s', strtotime( '-7 days' ) )
			)
		);

		return $stats;
	}

	/**
	 * Retry failed job
	 *
	 * @param int $job_id Job ID.
	 * @return bool|WP_Error Success or error.
	 */
	public function retry_job( $job_id ) {
		global $wpdb;

		// Get job.
		$job = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}vidtoarticle_jobs WHERE id = %d",
				$job_id
			)
		);

		if ( ! $job ) {
			return new WP_Error( 'not_found', __( 'Job not found', 'vidtoarticle-publisher' ) );
		}

		// Check if already published.
		if ( 'published' === $job->status ) {
			return new WP_Error( 'already_published', __( 'Job already published', 'vidtoarticle-publisher' ) );
		}

		// Request retry from backend.
		$api_client = new VidToArticle_API_Client();
		$response   = $api_client->retry_job( $job->backend_job_id );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Update local status.
		$wpdb->update(
			$wpdb->prefix . 'vidtoarticle_jobs',
			array(
				'status'      => 'pending',
				'retry_count' => intval( $job->retry_count ) + 1,
			),
			array( 'id' => $job_id ),
			array( '%s', '%d' ),
			array( '%d' )
		);

		return true;
	}

	/**
	 * Cancel pending job
	 *
	 * @param int $job_id Job ID.
	 * @return bool|WP_Error Success or error.
	 */
	public function cancel_job( $job_id ) {
		global $wpdb;

		// Get job.
		$job = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}vidtoarticle_jobs WHERE id = %d",
				$job_id
			)
		);

		if ( ! $job ) {
			return new WP_Error( 'not_found', __( 'Job not found', 'vidtoarticle-publisher' ) );
		}

		// Check if already published.
		if ( 'published' === $job->status ) {
			return new WP_Error( 'already_published', __( 'Cannot cancel published job', 'vidtoarticle-publisher' ) );
		}

		// Update local status to cancelled.
		$wpdb->update(
			$wpdb->prefix . 'vidtoarticle_jobs',
			array(
				'status'         => 'cancelled',
				'failure_reason' => __( 'Cancelled by user', 'vidtoarticle-publisher' ),
			),
			array( 'id' => $job_id ),
			array( '%s', '%s' ),
			array( '%d' )
		);

		// Log activity.
		VidToArticle_Logger::log(
			'job_cancelled',
			sprintf( __( 'Job #%d cancelled by user', 'vidtoarticle-publisher' ), $job_id ),
			array(
				'job_id'    => $job_id,
				'video_id'  => $job->video_id,
			)
		);

		return true;
	}
}
