<?php
/**
 * Post Publisher
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Post Publisher Class
 */
class VidToArticle_Post_Publisher {

	/**
	 * Publish article as WordPress post
	 *
	 * @param array  $article  Article data from backend.
	 * @param array  $video    Video metadata.
	 * @param string $job_id   Backend job ID.
	 * @param array  $metadata Article metadata (separated from content).
	 * @return array|WP_Error Post data or error.
	 */
	public function publish_article( $article, $video, $job_id, $metadata = array() ) {
		global $wpdb;

		// Get job from database to find source settings.
		$job = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT j.*, s.wp_category, s.wp_author
				FROM {$wpdb->prefix}vidtoarticle_jobs j
				LEFT JOIN {$wpdb->prefix}vidtoarticle_sources s ON j.source_id = s.id
				WHERE j.backend_job_id = %s",
				$job_id
			)
		);

		// CRITICAL: Check if job already has a post_id (prevents duplicate publishing).
		if ( $job && $job->post_id ) {
			// Job already published - return existing post info.
			return array(
				'post_id'     => $job->post_id,
				'post_status' => get_post_status( $job->post_id ),
				'post_url'    => get_permalink( $job->post_id ),
				'already_published' => true,
			);
		}

		// CRITICAL: Video-level deduplication — prevent duplicate posts for the same video
		// even if webhooks are delivered multiple times with different job IDs.
		if ( isset( $video['id'] ) && ! empty( $video['id'] ) ) {
			$existing_posts = get_posts( array(
				'post_type'   => 'post',
				'post_status' => array( 'publish', 'draft', 'pending', 'private' ),
				'meta_key'    => '_vidtoarticle_video_id',
				'meta_value'  => sanitize_text_field( $video['id'] ),
				'numberposts' => 1,
				'fields'      => 'ids',
			) );

			if ( ! empty( $existing_posts ) ) {
				$existing_post_id = $existing_posts[0];
				return array(
					'post_id'           => $existing_post_id,
					'post_status'       => get_post_status( $existing_post_id ),
					'post_url'          => get_permalink( $existing_post_id ),
					'already_published' => true,
				);
			}
		}

		// If job doesn't exist, create it.
		if ( ! $job && isset( $video['id'] ) ) {
			// Try to find matching source.
			$source = $this->find_matching_source( $video['id'] );

			if ( $source ) {
				// Create job record.
				$wpdb->insert(
					$wpdb->prefix . 'vidtoarticle_jobs',
					array(
						'source_id'       => $source->id,
						'backend_job_id'  => $job_id,
						'video_id'        => $video['id'],
						'video_url'       => $video['url'] ?? '',
						'video_title'     => $video['title'] ?? '',
						'status'          => 'processing',
						'created_at'      => current_time( 'mysql' ),
					),
					array( '%d', '%s', '%s', '%s', '%s', '%s', '%s' )
				);

				$job_db_id = $wpdb->insert_id;

				// Re-fetch job.
				$job = $wpdb->get_row(
					$wpdb->prepare(
						"SELECT j.*, s.wp_category, s.wp_author
						FROM {$wpdb->prefix}vidtoarticle_jobs j
						LEFT JOIN {$wpdb->prefix}vidtoarticle_sources s ON j.source_id = s.id
						WHERE j.id = %d",
						$job_db_id
					)
				);
			}
		}

		// Determine post status (default to draft for review).
		$post_status = get_option( 'vidtoarticle_default_post_status', 'draft' );

		// Strip leading H1 title from content to prevent duplication with post_title.
		$content = $article['content'];
		$content = preg_replace( '/^\s*#\s+.+\n*/m', '', $content, 1 );
		$content = ltrim( $content, "\n" );

		// Prepare post data.
		$post_data = array(
			'post_title'   => sanitize_text_field( $article['title'] ),
			'post_content' => wp_kses_post( $this->convert_markdown_to_html( $content ) ),
			'post_excerpt' => wp_kses_post( $article['excerpt'] ?? '' ),
			'post_status'  => $post_status,
			'post_type'    => 'post',
		);

		// Set category if specified.
		if ( ! empty( $metadata['wp_category_id'] ) ) {
			$post_data['post_category'] = array( intval( $metadata['wp_category_id'] ) );
		} elseif ( $job && $job->wp_category ) {
			$post_data['post_category'] = array( intval( $job->wp_category ) );
		} elseif ( ! empty( $article['suggested_categories'] ) ) {
			// Use suggested categories.
			$categories = array();
			foreach ( $article['suggested_categories'] as $cat_name ) {
				$cat = get_category_by_slug( sanitize_title( $cat_name ) );
				if ( $cat ) {
					$categories[] = $cat->term_id;
				}
			}
			if ( ! empty( $categories ) ) {
				$post_data['post_category'] = $categories;
			}
		}

		// Set author if specified.
		if ( ! empty( $metadata['wp_author_id'] ) ) {
			$post_data['post_author'] = intval( $metadata['wp_author_id'] );
		} elseif ( $job && $job->wp_author ) {
			$post_data['post_author'] = intval( $job->wp_author );
		}

		// Create the post.
		$post_id = wp_insert_post( $post_data, true );

		if ( is_wp_error( $post_id ) ) {
			// Update job status.
			if ( $job ) {
				$wpdb->update(
					$wpdb->prefix . 'vidtoarticle_jobs',
					array(
						'status'         => 'failed',
						'failure_reason' => $post_id->get_error_message(),
						'retry_count'    => intval( $job->retry_count ) + 1,
					),
					array( 'id' => $job->id ),
					array( '%s', '%s', '%d' ),
					array( '%d' )
				);
			}

			return $post_id;
		}

		// Add tags.
		if ( ! empty( $article['tags'] ) ) {
			wp_set_post_tags( $post_id, $article['tags'], false );
		}

		// Add custom fields using structured metadata.
		if ( isset( $video['url'] ) ) {
			update_post_meta( $post_id, '_vidtoarticle_video_url', esc_url_raw( $video['url'] ) );
		}

		if ( isset( $video['id'] ) ) {
			update_post_meta( $post_id, '_vidtoarticle_video_id', sanitize_text_field( $video['id'] ) );
		}

		if ( isset( $article['style'] ) ) {
			update_post_meta( $post_id, '_vidtoarticle_style', sanitize_text_field( $article['style'] ) );
		}

		if ( isset( $article['word_count'] ) ) {
			update_post_meta( $post_id, '_vidtoarticle_word_count', intval( $article['word_count'] ) );
		}

		if ( isset( $article['reading_time_minutes'] ) ) {
			update_post_meta( $post_id, '_vidtoarticle_reading_time', intval( $article['reading_time_minutes'] ) );
		}

		// Store separated metadata from backend.
		if ( ! empty( $metadata ) ) {
			// Video source URL from metadata.
			if ( isset( $metadata['source'] ) ) {
				update_post_meta( $post_id, '_vidtoarticle_source_url', esc_url_raw( $metadata['source'] ) );
			}

			// Video ID from metadata.
			if ( isset( $metadata['video_id'] ) ) {
				update_post_meta( $post_id, '_vidtoarticle_video_id_backend', sanitize_text_field( $metadata['video_id'] ) );
			}

			// Generation timestamp from metadata.
			if ( isset( $metadata['generated_at'] ) ) {
				update_post_meta( $post_id, '_vidtoarticle_generated_at_backend', sanitize_text_field( $metadata['generated_at'] ) );
			}

			// Generator info from metadata.
			if ( isset( $metadata['generator'] ) ) {
				update_post_meta( $post_id, '_vidtoarticle_generator', sanitize_text_field( $metadata['generator'] ) );
			}
		}

		// Handle featured image from API (Pexels/Unsplash) - priority over YouTube thumbnail
		$featured_image_handler = new VidToArticle_Featured_Image_Handler();
		$api_featured_image = $featured_image_handler->get_image_from_article( $article );

		if ( $api_featured_image ) {
			$featured_image_handler->set_featured_image( $post_id, $api_featured_image );
		} elseif ( isset( $video['thumbnail_url'] ) ) {
			// Fallback to YouTube thumbnail if no API image
			$this->set_featured_image( $post_id, $video['thumbnail_url'], $article['title'] );
		}

		// Update job status.
		if ( $job ) {
			$wpdb->update(
				$wpdb->prefix . 'vidtoarticle_jobs',
				array(
					'status'       => 'published',
					'post_id'      => $post_id,
					'processed_at' => current_time( 'mysql' ),
				),
				array( 'id' => $job->id ),
				array( '%s', '%d', '%s' ),
				array( '%d' )
			);
		}

		// Return success data.
		return array(
			'post_id'     => $post_id,
			'post_status' => $post_status,
			'post_url'    => get_permalink( $post_id ),
		);
	}

	/**
	 * Convert Markdown to HTML
	 *
	 * @param string $markdown Markdown content.
	 * @return string HTML content.
	 */
	private function convert_markdown_to_html( $markdown ) {
		// Strip any TAGS line that may have leaked through from the LLM output.
		$markdown = preg_replace( '/^[\*\-•]*\s*\**\s*tags?\s*[:\-–]\s*\**\s*.+$/mi', '', $markdown );

		// Use Parsedown (bundled with this plugin) for proper markdown-to-HTML conversion.
		if ( class_exists( 'Parsedown' ) ) {
			$parsedown = new Parsedown();
			$parsedown->setSafeMode( true );
			return $parsedown->text( $markdown );
		}

		// Minimal fallback (should never reach here since Parsedown is bundled).
		$html = $markdown;
		$html = preg_replace( '/^###### (.+)$/m', '<h6>$1</h6>', $html );
		$html = preg_replace( '/^##### (.+)$/m', '<h5>$1</h5>', $html );
		$html = preg_replace( '/^#### (.+)$/m', '<h4>$1</h4>', $html );
		$html = preg_replace( '/^### (.+)$/m', '<h3>$1</h3>', $html );
		$html = preg_replace( '/^## (.+)$/m', '<h2>$1</h2>', $html );
		$html = preg_replace( '/^# (.+)$/m', '<h1>$1</h1>', $html );
		$html = preg_replace( '/\*\*\*(.+?)\*\*\*/', '<strong><em>$1</em></strong>', $html );
		$html = preg_replace( '/\*\*(.+?)\*\*/', '<strong>$1</strong>', $html );
		$html = preg_replace( '/\*([^\*\n]+?)\*/', '<em>$1</em>', $html );
		$html = preg_replace( '/\[(.+?)\]\((.+?)\)/', '<a href="$2">$1</a>', $html );
		$html = wpautop( $html );
		return $html;
	}

	/**
	 * Set featured image from URL
	 *
	 * @param int    $post_id    Post ID.
	 * @param string $image_url  Image URL.
	 * @param string $image_name Image name/alt text.
	 * @return int|false Attachment ID or false on failure.
	 */
	private function set_featured_image( $post_id, $image_url, $image_name ) {
		// Require image functions.
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/media.php';
		require_once ABSPATH . 'wp-admin/includes/image.php';

		// Download image.
		$tmp = download_url( $image_url );

		if ( is_wp_error( $tmp ) ) {
			return false;
		}

		// Get file extension.
		$ext      = pathinfo( $image_url, PATHINFO_EXTENSION );
		$filename = sanitize_file_name( $image_name ) . '.' . $ext;

		// Prepare file array.
		$file_array = array(
			'name'     => $filename,
			'tmp_name' => $tmp,
		);

		// Upload to media library.
		$attachment_id = media_handle_sideload( $file_array, $post_id, $image_name );

		// Delete temp file.
		@unlink( $tmp );

		if ( is_wp_error( $attachment_id ) ) {
			return false;
		}

		// Set as featured image.
		set_post_thumbnail( $post_id, $attachment_id );

		return $attachment_id;
	}

	/**
	 * Find matching source for video
	 *
	 * @param string $video_id YouTube video ID.
	 * @return object|null Source object or null.
	 */
	private function find_matching_source( $video_id ) {
		global $wpdb;

		// This is a simplified version - in reality, we'd need to check
		// if the video belongs to any of the monitored playlists/channels.
		// For now, just get the first active source.
		return $wpdb->get_row(
			"SELECT * FROM {$wpdb->prefix}vidtoarticle_sources
			WHERE is_active = 1
			ORDER BY created_at ASC
			LIMIT 1"
		);
	}
}
