<?php
/**
 * Logging Handler
 *
 * Provides centralized logging functionality for the plugin.
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Logger Class
 *
 * Handles all plugin logging with proper error levels and context.
 */
class VidToArticle_Logger {

	/**
	 * Log levels
	 */
	const LEVEL_ERROR   = 'error';
	const LEVEL_WARNING = 'warning';
	const LEVEL_INFO    = 'info';
	const LEVEL_DEBUG   = 'debug';

	/**
	 * Single instance
	 *
	 * @var VidToArticle_Logger
	 */
	private static $instance = null;

	/**
	 * Get instance
	 *
	 * @return VidToArticle_Logger
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Log error message
	 *
	 * @param string $message Message to log.
	 * @param array  $context Additional context.
	 */
	public static function error( $message, $context = array() ) {
		self::log( self::LEVEL_ERROR, $message, $context );
	}

	/**
	 * Log warning message
	 *
	 * @param string $message Message to log.
	 * @param array  $context Additional context.
	 */
	public static function warning( $message, $context = array() ) {
		self::log( self::LEVEL_WARNING, $message, $context );
	}

	/**
	 * Log info message
	 *
	 * @param string $message Message to log.
	 * @param array  $context Additional context.
	 */
	public static function info( $message, $context = array() ) {
		self::log( self::LEVEL_INFO, $message, $context );
	}

	/**
	 * Log debug message
	 *
	 * @param string $message Message to log.
	 * @param array  $context Additional context.
	 */
	public static function debug( $message, $context = array() ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			self::log( self::LEVEL_DEBUG, $message, $context );
		}
	}

	/**
	 * Log message
	 *
	 * @param string $level   Log level.
	 * @param string $message Message to log.
	 * @param array  $context Additional context.
	 */
	private static function log( $level, $message, $context = array() ) {
		global $wpdb;

		// Sanitize message.
		$message = sanitize_text_field( $message );

		// Log to WordPress error log if WP_DEBUG_LOG is enabled.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( sprintf( '[VidToArticle][%s] %s', strtoupper( $level ), $message ) );
		}

		// Log to database activity table.
		$wpdb->insert(
			$wpdb->prefix . 'vidtoarticle_activity',
			array(
				'action'     => 'log_' . $level,
				'message'    => $message,
				'context'    => ! empty( $context ) ? wp_json_encode( $context ) : null,
				'created_at' => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%s', '%s' )
		);
	}

	/**
	 * Log activity (for backward compatibility)
	 *
	 * @param string $action  Action type.
	 * @param string $message Log message.
	 * @param array  $context Additional context.
	 */
	public static function log_activity( $action, $message, $context = array() ) {
		global $wpdb;

		$wpdb->insert(
			$wpdb->prefix . 'vidtoarticle_activity',
			array(
				'action'     => $action,
				'message'    => $message,
				'context'    => ! empty( $context ) ? wp_json_encode( $context ) : null,
				'created_at' => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%s', '%s' )
		);
	}
}
