<?php
/**
 * Plugin Autoloader
 *
 * Handles automatic class loading following PSR-4 standards adapted for WordPress.
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Autoloader Class
 */
class VidToArticle_Loader {

	/**
	 * Class map for faster loading
	 *
	 * @var array
	 */
	private static $class_map = array(
		'VidToArticle_API_Client'           => 'includes/class-api-client.php',
		'VidToArticle_OAuth_Handler'        => 'includes/class-oauth-handler.php',
		'VidToArticle_Webhook_Listener'     => 'includes/class-webhook-listener.php',
		'VidToArticle_Fallback_Poller'      => 'includes/class-fallback-poller.php',
		'VidToArticle_Post_Publisher'       => 'includes/class-post-publisher.php',
		'VidToArticle_Featured_Image_Handler' => 'includes/class-featured-image-handler.php',
		'VidToArticle_Source_Manager'       => 'includes/class-source-manager.php',
		'VidToArticle_Queue_Manager'        => 'includes/class-queue-manager.php',
		'VidToArticle_API_Handler'          => 'includes/class-api-handler.php',
		'VidToArticle_Admin'                => 'admin/class-admin.php',
		'VidToArticle_REST_Controller'      => 'includes/class-rest-controller.php',
	);

	/**
	 * Register autoloader
	 */
	public static function register() {
		spl_autoload_register( array( __CLASS__, 'autoload' ) );
	}

	/**
	 * Autoload classes
	 *
	 * @param string $class Class name.
	 */
	public static function autoload( $class ) {
		// Check if this is a VidToArticle class.
		if ( 0 !== strpos( $class, 'VidToArticle_' ) ) {
			return;
		}

		// Check class map first (faster).
		if ( isset( self::$class_map[ $class ] ) ) {
			$file = VIDTOARTICLE_PLUGIN_DIR . self::$class_map[ $class ];
			if ( file_exists( $file ) ) {
				require_once $file;
				return;
			}
		}

		// Fallback to PSR-4 style loading.
		$class_file = 'class-' . str_replace( '_', '-', strtolower( $class ) ) . '.php';

		// Try includes directory.
		$file = VIDTOARTICLE_PLUGIN_DIR . 'includes/' . $class_file;
		if ( file_exists( $file ) ) {
			require_once $file;
			return;
		}

		// Try admin directory.
		$file = VIDTOARTICLE_PLUGIN_DIR . 'admin/' . $class_file;
		if ( file_exists( $file ) ) {
			require_once $file;
		}
	}

	/**
	 * Load all required files
	 */
	public static function load_dependencies() {
		// Load core classes that don't follow autoload pattern.
		$files = array(
			'includes/class-api-client.php',
			'includes/class-oauth-handler.php',
			'includes/class-webhook-listener.php',
			'includes/class-fallback-poller.php',
			'includes/class-post-publisher.php',
			'includes/class-featured-image-handler.php',
			'includes/class-source-manager.php',
			'includes/class-queue-manager.php',
			'includes/class-api-handler.php',
		);

		foreach ( $files as $file ) {
			$filepath = VIDTOARTICLE_PLUGIN_DIR . $file;
			if ( file_exists( $filepath ) ) {
				require_once $filepath;
			}
		}
	}
}
