<?php
/**
 * Featured Image Handler
 * Downloads and sets featured images from Pexels/Unsplash API data
 * Handles proper attribution and metadata storage
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Featured Image Handler Class
 */
class VidToArticle_Featured_Image_Handler {

	/**
	 * Download and set featured image for post from API data
	 *
	 * @param int   $post_id    Post ID.
	 * @param array $image_data Image metadata from VidToArticle API.
	 * @return int|false Attachment ID on success, false on failure.
	 */
	public function set_featured_image( $post_id, $image_data ) {
		if ( empty( $image_data ) || ! is_array( $image_data ) ) {
			VidToArticle_Logger::log( 'No valid image data provided for featured image', array(
				'post_id' => $post_id,
				'image_data' => $image_data,
			) );
			return false;
		}

		// Check if post already has featured image
		if ( has_post_thumbnail( $post_id ) ) {
			VidToArticle_Logger::log( 'Post already has featured image, skipping', array(
				'post_id' => $post_id,
			) );
			return false;
		}

		// Validate required image data fields
		if ( empty( $image_data['url'] ) ) {
			VidToArticle_Logger::log( 'Missing image URL in image data', array(
				'post_id' => $post_id,
				'image_data' => $image_data,
			) );
			return false;
		}

		// Load WordPress file handling functions
		require_once ABSPATH . 'wp-admin/includes/media.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/image.php';

		// Use download URL if available, otherwise use regular URL
		$image_url = ! empty( $image_data['downloadUrl'] ) ? $image_data['downloadUrl'] : $image_data['url'];

		// Validate URL
		if ( ! filter_var( $image_url, FILTER_VALIDATE_URL ) ) {
			VidToArticle_Logger::log( 'Invalid image URL provided', array(
				'post_id' => $post_id,
				'image_url' => $image_url,
			) );
			return false;
		}

		// Download image with timeout
		$tmp_file = $this->download_image_with_timeout( $image_url );

		if ( is_wp_error( $tmp_file ) ) {
			VidToArticle_Logger::log( 'Failed to download image', array(
				'error' => $tmp_file->get_error_message(),
				'url'   => $image_url,
				'post_id' => $post_id,
			) );
			return false;
		}

		// Prepare file array for sideload
		$file_array = array(
			'name'     => $this->generate_filename( $image_data, $post_id ),
			'tmp_name' => $tmp_file,
		);

		// Upload to media library
		$attachment_id = media_handle_sideload( $file_array, $post_id );

		// Clean up temp file
		if ( file_exists( $tmp_file ) ) {
			@unlink( $tmp_file );
		}

		if ( is_wp_error( $attachment_id ) ) {
			VidToArticle_Logger::log( 'Failed to create attachment', array(
				'error'   => $attachment_id->get_error_message(),
				'post_id' => $post_id,
				'image_url' => $image_url,
			) );
			return false;
		}

		// Set image metadata (alt text, description, attribution)
		$this->set_image_metadata( $attachment_id, $image_data );

		// Set as featured image
		$result = set_post_thumbnail( $post_id, $attachment_id );

		if ( $result ) {
			VidToArticle_Logger::log( 'Featured image set successfully', array(
				'post_id'       => $post_id,
				'attachment_id' => $attachment_id,
				'source'        => $image_data['source'] ?? 'unknown',
				'photographer'  => $image_data['photographer'] ?? 'unknown',
			) );

			// Update post meta to track image source
			update_post_meta( $post_id, '_vidtoarticle_featured_image_source', $image_data['source'] ?? 'unknown' );
			update_post_meta( $post_id, '_vidtoarticle_featured_image_set_at', current_time( 'mysql' ) );
		} else {
			VidToArticle_Logger::log( 'Failed to set featured image', array(
				'post_id' => $post_id,
				'attachment_id' => $attachment_id,
			) );
		}

		return $attachment_id;
	}

	/**
	 * Download image with timeout and error handling
	 *
	 * @param string $url Image URL to download.
	 * @return string|WP_Error Temporary file path or error.
	 */
	private function download_image_with_timeout( $url ) {
		// Use WordPress built-in download function with timeout
		$timeout = 30; // 30 seconds timeout

		// Download the file
		$tmp_file = download_url( $url, $timeout );

		if ( is_wp_error( $tmp_file ) ) {
			return $tmp_file;
		}

		// Validate downloaded file
		if ( ! file_exists( $tmp_file ) ) {
			return new WP_Error( 'file_not_found', 'Downloaded file not found' );
		}

		// Check file size (max 10MB)
		$file_size = filesize( $tmp_file );
		if ( $file_size > 10 * 1024 * 1024 ) {
			unlink( $tmp_file );
			return new WP_Error( 'file_too_large', 'Image file is too large (max 10MB)' );
		}

		// Check if it's actually an image
		$image_info = @getimagesize( $tmp_file );
		if ( ! $image_info ) {
			unlink( $tmp_file );
			return new WP_Error( 'invalid_image', 'Downloaded file is not a valid image' );
		}

		return $tmp_file;
	}

	/**
	 * Generate filename for image
	 *
	 * @param array $image_data Image metadata.
	 * @param int   $post_id    Post ID.
	 * @return string Filename.
	 */
	private function generate_filename( $image_data, $post_id ) {
		$source = isset( $image_data['source'] ) ? sanitize_title( $image_data['source'] ) : 'image';
		$alt    = isset( $image_data['alt'] ) ? sanitize_title( $image_data['alt'] ) : 'featured';

		// Limit length and clean up
		$alt = substr( $alt, 0, 50 );
		$alt = preg_replace( '/[^a-z0-9\-]/', '-', $alt );
		$alt = preg_replace( '/-+/', '-', $alt );
		$alt = trim( $alt, '-' );

		// If alt is empty after sanitization, use a default
		if ( empty( $alt ) ) {
			$alt = 'featured-image';
		}

		return sprintf( '%s-%s-%d.jpg', $source, $alt, $post_id );
	}

	/**
	 * Set image metadata (alt text, description, attribution)
	 *
	 * @param int   $attachment_id Attachment ID.
	 * @param array $image_data    Image metadata.
	 */
	private function set_image_metadata( $attachment_id, $image_data ) {
		// Set alt text
		if ( ! empty( $image_data['alt'] ) ) {
			update_post_meta( $attachment_id, '_wp_attachment_image_alt', sanitize_text_field( $image_data['alt'] ) );
		}

		// Set image caption with photographer attribution
		if ( ! empty( $image_data['photographer'] ) ) {
			$photographer = sanitize_text_field( $image_data['photographer'] );
			$source = ! empty( $image_data['source'] ) ? ucfirst( sanitize_text_field( $image_data['source'] ) ) : 'Unknown';

			$caption = sprintf(
				'Photo by %s on %s',
				$photographer,
				$source
			);

			// Add photographer URL if available
			if ( ! empty( $image_data['photographerUrl'] ) ) {
				$photographer_url = esc_url( $image_data['photographerUrl'] );
				$caption = sprintf(
					'Photo by <a href="%s" target="_blank" rel="noopener noreferrer">%s</a> on %s',
					$photographer_url,
					$photographer,
					$source
				);
			}

			wp_update_post( array(
				'ID'           => $attachment_id,
				'post_excerpt' => $caption,
				'post_content' => $caption, // Also set as description
			) );
		}

		// Store source metadata for tracking
		if ( ! empty( $image_data['source'] ) ) {
			update_post_meta( $attachment_id, '_vidtoarticle_image_source', sanitize_text_field( $image_data['source'] ) );
		}

		if ( ! empty( $image_data['photographer'] ) ) {
			update_post_meta( $attachment_id, '_vidtoarticle_photographer', sanitize_text_field( $image_data['photographer'] ) );
		}

		if ( ! empty( $image_data['photographerUrl'] ) ) {
			update_post_meta( $attachment_id, '_vidtoarticle_photographer_url', esc_url_raw( $image_data['photographerUrl'] ) );
		}

		if ( ! empty( $image_data['url'] ) ) {
			update_post_meta( $attachment_id, '_vidtoarticle_original_url', esc_url_raw( $image_data['url'] ) );
		}

		if ( ! empty( $image_data['downloadUrl'] ) ) {
			update_post_meta( $attachment_id, '_vidtoarticle_download_url', esc_url_raw( $image_data['downloadUrl'] ) );
		}
	}

	/**
	 * Check if featured image handling is enabled
	 *
	 * @return bool True if enabled
	 */
	public function is_enabled() {
		return true; // Always enabled, will skip gracefully if no image data
	}

	/**
	 * Get image source from article data
	 *
	 * @param array $article Article data from API.
	 * @return array|null Image data or null if not found
	 */
	public function get_image_from_article( $article ) {
		if ( ! is_array( $article ) ) {
			return null;
		}

		// Check for featuredImage in article data
		if ( ! empty( $article['featuredImage'] ) && is_array( $article['featuredImage'] ) ) {
			return $article['featuredImage'];
		}

		// Check nested structure if applicable
		if ( ! empty( $article['data']['featuredImage'] ) && is_array( $article['data']['featuredImage'] ) ) {
			return $article['data']['featuredImage'];
		}

		return null;
	}
}