<?php
/**
 * Fallback Poller (Safety Net)
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Fallback Poller Class
 */
class VidToArticle_Fallback_Poller {

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'vidtoarticle_fallback_poll', array( $this, 'poll_missed_jobs' ) );
	}

	/**
	 * Poll for missed jobs
	 */
	public function poll_missed_jobs() {
		$api_client = new VidToArticle_API_Client();

		// Check if connected.
		if ( ! $api_client->is_connected() ) {
			return;
		}

		// Get missed jobs from backend.
		$response = $api_client->get_missed_jobs();

		if ( is_wp_error( $response ) ) {
			$this->log_activity( 'fallback_poll_failed', $response->get_error_message() );
			return;
		}

		if ( ! isset( $response['success'] ) || ! $response['success'] ) {
			return;
		}

		$missed_jobs = $response['data']['jobs'] ?? array();

		if ( empty( $missed_jobs ) ) {
			$this->log_activity( 'fallback_poll_complete', __( 'No missed jobs found', 'vidtoarticle-publisher' ) );
			return;
		}

		// Process each missed job.
		$publisher = new VidToArticle_Post_Publisher();
		$processed = 0;
		$failed    = 0;

		foreach ( $missed_jobs as $job_data ) {
			$job_id  = $job_data['job_id'] ?? '';
			$article = $job_data['article'] ?? array();
			$video   = $job_data['video'] ?? array();

			if ( empty( $job_id ) || empty( $article ) ) {
				continue;
			}

			// Publish article.
			$result = $publisher->publish_article( $article, $video, $job_id );

			if ( is_wp_error( $result ) ) {
				$failed++;
				$this->log_activity( 'fallback_publish_failed', $result->get_error_message(), array(
					'job_id' => $job_id,
				) );
				continue;
			}

			$processed++;

			// Confirm with backend.
			$api_client->confirm_webhook( $job_id );

			// Check if this was already published (duplicate detection).
			if ( isset( $result['already_published'] ) && $result['already_published'] ) {
				$this->log_activity( 'fallback_duplicate_detected', __( 'Article was already published, skipped duplicate', 'vidtoarticle-publisher' ), array(
					'job_id'  => $job_id,
					'post_id' => $result['post_id'],
				) );
			} else {
				$this->log_activity( 'fallback_published', __( 'Article published via fallback polling', 'vidtoarticle-publisher' ), array(
					'job_id'  => $job_id,
					'post_id' => $result['post_id'],
				) );
			}
		}

		// Log summary.
		$this->log_activity( 'fallback_poll_complete', sprintf(
			/* translators: 1: processed count, 2: failed count */
			__( 'Fallback polling complete: %1$d processed, %2$d failed', 'vidtoarticle-publisher' ),
			$processed,
			$failed
		) );
	}

	/**
	 * Log activity
	 *
	 * @param string $action  Action type.
	 * @param string $message Log message.
	 * @param array  $context Additional context.
	 */
	private function log_activity( $action, $message, $context = array() ) {
		global $wpdb;

		$wpdb->insert(
			$wpdb->prefix . 'vidtoarticle_activity',
			array(
				'action'     => $action,
				'message'    => $message,
				'context'    => ! empty( $context ) ? wp_json_encode( $context ) : null,
				'created_at' => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%s', '%s' )
		);
	}
}
