<?php
/**
 * API Client with HMAC Authentication
 *
 * @package VidToArticle_Publisher
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * API Client Class
 */
class VidToArticle_API_Client {

	/**
	 * API Base URL
	 *
	 * @var string
	 */
	private $api_base_url;

	/**
	 * API Secret Key
	 *
	 * @var string|null
	 */
	private $secret_key;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->api_base_url = defined( 'VIDTOARTICLE_API_URL' )
			? VIDTOARTICLE_API_URL
			: 'https://api.vidtoarticle.com';

		$this->secret_key = get_option( 'vidtoarticle_secret_key' );
	}

	/**
	 * Check if plugin is connected
	 *
	 * @return bool
	 */
	public function is_connected() {
		return ! empty( $this->secret_key );
	}

	/**
	 * Generate HMAC signature for request
	 *
	 * @param string $method HTTP method.
	 * @param string $path   API path.
	 * @param string $body   Request body (JSON string).
	 * @param int    $timestamp Unix timestamp.
	 * @return string HMAC signature.
	 */
	private function generate_signature( $method, $path, $body = '', $timestamp = null ) {
		if ( empty( $this->secret_key ) ) {
			return '';
		}

		if ( null === $timestamp ) {
			$timestamp = time();
		}

		// Create body hash.
		$body_hash = hash( 'sha256', $body );

		// Create signature string.
		$signature_string = $timestamp . $method . $path . $body_hash;

		// Generate HMAC signature.
		return hash_hmac( 'sha256', $signature_string, $this->secret_key );
	}

	/**
	 * Make authenticated API request
	 *
	 * @param string $method   HTTP method (GET, POST, PUT, DELETE).
	 * @param string $endpoint API endpoint path.
	 * @param array  $data     Request data.
	 * @return array|WP_Error Response data or error.
	 */
	public function request( $method, $endpoint, $data = array() ) {
		$url  = $this->api_base_url . $endpoint;
		$body = '';

		// Prepare request body for POST/PUT.
		if ( in_array( $method, array( 'POST', 'PUT' ), true ) && ! empty( $data ) ) {
			$body = wp_json_encode( $data );
		}

		// Generate HMAC signature.
		$timestamp = time();
		$signature = $this->generate_signature( $method, $endpoint, $body, $timestamp );

		// Prepare headers.
		$headers = array(
			'Content-Type'              => 'application/json',
			'X-VidToArticle-Timestamp'  => $timestamp,
			'X-VidToArticle-Signature'  => $signature,
			'X-VidToArticle-Site-URL'   => get_site_url(),
		);

		// Prepare request args.
		$args = array(
			'method'  => $method,
			'headers' => $headers,
			'timeout' => 30,
		);

		if ( ! empty( $body ) ) {
			$args['body'] = $body;
		}

		// Add query params for GET requests.
		if ( 'GET' === $method && ! empty( $data ) ) {
			$url = add_query_arg( $data, $url );
		}

		// Make request.
		$response = wp_remote_request( $url, $args );

		// Handle errors.
		if ( is_wp_error( $response ) ) {
			$this->log_error( 'API Request Failed', array(
				'endpoint' => $endpoint,
				'error'    => $response->get_error_message(),
			) );
			return $response;
		}

		// Get response code and body.
		$code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );

		// Parse JSON response.
		$decoded = json_decode( $body, true );

		// Check for non-200 responses.
		if ( $code < 200 || $code >= 300 ) {
			$error_message = isset( $decoded['message'] ) ? $decoded['message'] : 'Unknown error';
			$error_details = isset( $decoded['error'] ) ? $decoded['error'] : null;

			$this->log_error(
				sprintf( 'API Error: %s (HTTP %d)', $error_details ? $error_details : 'Request failed', $code ),
				array(
					'endpoint'    => $endpoint,
					'method'      => $method,
					'status_code' => $code,
					'error_type'  => $error_details,
					'message'     => $error_message,
					'response'    => is_array( $decoded ) ? wp_json_encode( $decoded ) : substr( $body, 0, 200 ),
				)
			);
			return new WP_Error( 'api_error', $error_message, array( 'status' => $code ) );
		}

		return $decoded;
	}

	/**
	 * Get connection status and quota
	 *
	 * @return array|WP_Error
	 */
	public function get_connection_status() {
		return $this->request( 'GET', '/api/wordpress/connect/status' );
	}

	/**
	 * Get monitored sources
	 *
	 * @return array|WP_Error
	 */
	public function get_sources() {
		return $this->request( 'GET', '/api/wordpress/sources' );
	}

	/**
	 * Create new monitored source
	 *
	 * @param array $source_data Source configuration.
	 * @return array|WP_Error
	 */
	public function create_source( $source_data ) {
		return $this->request( 'POST', '/api/wordpress/sources', $source_data );
	}

	/**
	 * Update monitored source
	 *
	 * @param string $source_id   Backend source ID.
	 * @param array  $source_data Updated configuration.
	 * @return array|WP_Error
	 */
	public function update_source( $source_id, $source_data ) {
		return $this->request( 'PUT', "/api/wordpress/sources/{$source_id}", $source_data );
	}

	/**
	 * Delete monitored source
	 *
	 * @param string $source_id Backend source ID.
	 * @return array|WP_Error
	 */
	public function delete_source( $source_id ) {
		return $this->request( 'DELETE', "/api/wordpress/sources/{$source_id}" );
	}

	/**
	 * Get missed jobs (fallback polling)
	 *
	 * @return array|WP_Error
	 */
	public function get_missed_jobs() {
		return $this->request( 'GET', '/api/wordpress/missed-jobs' );
	}

	/**
	 * Confirm webhook receipt
	 *
	 * @param string $job_id Backend job ID.
	 * @return array|WP_Error
	 */
	public function confirm_webhook( $job_id ) {
		return $this->request( 'POST', '/api/wordpress/webhook/confirm', array( 'job_id' => $job_id ) );
	}

	/**
	 * Retry failed job
	 *
	 * @param string $job_id Backend job ID.
	 * @return array|WP_Error
	 */
	public function retry_job( $job_id ) {
		return $this->request( 'POST', "/api/wordpress/jobs/{$job_id}/retry" );
	}

	/**
	 * Get backend logs (webhook deliveries, job processing history)
	 *
	 * @param array $params Query parameters (level, type, limit).
	 * @return array|WP_Error
	 */
	public function get_backend_logs( $params = array() ) {
		return $this->request( 'GET', '/api/wordpress/logs', $params );
	}

	/**
	 * Exchange OAuth token for secret key
	 * Note: This is an unauthenticated request since we don't have the secret key yet
	 *
	 * @param string $token One-time OAuth token.
	 * @return array|WP_Error
	 */
	public function exchange_token( $token ) {
		$url = $this->api_base_url . '/api/wordpress/connect/exchange';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type' => 'application/json',
			),
			'body'    => wp_json_encode( array( 'token' => $token ) ),
			'timeout' => 30,
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			$this->log_error( 'Token Exchange Failed', array(
				'error' => $response->get_error_message(),
			) );
			return $response;
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );
		$decoded = json_decode( $body, true );

		if ( $code < 200 || $code >= 300 ) {
			$error_message = isset( $decoded['message'] ) ? $decoded['message'] : 'Unknown error';
			$this->log_error( 'Token Exchange Error', array(
				'code'    => $code,
				'message' => $error_message,
			) );
			return new WP_Error( 'exchange_error', $error_message, array( 'status' => $code ) );
		}

		return array( 'success' => true, 'data' => $decoded );
	}

	/**
	 * Revoke API connection
	 *
	 * @return array|WP_Error
	 */
	public function revoke_connection() {
		// Make the revoke request BEFORE deleting the secret key (need it for HMAC).
		$response = $this->request( 'DELETE', '/api/wordpress/connect/revoke' );

		// Clear local secret key regardless of API response.
		delete_option( 'vidtoarticle_secret_key' );
		delete_option( 'vidtoarticle_connection_status' );
		$this->secret_key = null;

		return $response;
	}

	/**
	 * Save secret key
	 *
	 * @param string $secret_key API secret key.
	 */
	public function save_secret_key( $secret_key ) {
		update_option( 'vidtoarticle_secret_key', $secret_key, false ); // Don't autoload.
		$this->secret_key = $secret_key;
	}

	/**
	 * Log error to activity log
	 *
	 * @param string $message Error message.
	 * @param array  $context Additional context.
	 */
	private function log_error( $message, $context = array() ) {
		global $wpdb;

		$wpdb->insert(
			$wpdb->prefix . 'vidtoarticle_activity',
			array(
				'action'     => 'api_error',
				'message'    => $message,
				'context'    => wp_json_encode( $context ),
				'created_at' => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%s', '%s' )
		);
	}
}
