/**
 * Connection Tab Component
 *
 * Manages the OAuth connection between WordPress and VidToArticle.com.
 * All source configuration and article generation happens on VidToArticle.com.
 *
 * @package VidToArticle_Publisher
 * @since 1.3.0 Simplified interface
 */

(function () {
  "use strict";

  const { createElement: el, Fragment, useState } = wp.element;
  const { Button, Card, CardBody, Notice } = wp.components;
  const { __ } = wp.i18n;
  const apiFetch = wp.apiFetch;

  // Dashboard URL from localized script or fallback
  const DASHBOARD_URL = window.vidToArticleAdmin?.dashboardUrl || "https://www.vidtoarticle.com/wordpress-connect";

  /**
   * Connection Tab - Handles VidToArticle.com OAuth connection
   */
  function ConnectionTab({ connectionStatus, refreshConnection, addNotice }) {
    const [connecting, setConnecting] = useState(false);
    const [disconnecting, setDisconnecting] = useState(false);

    /**
     * Initiate OAuth connection flow (redirect-based)
     */
    const handleConnect = async () => {
      setConnecting(true);
      try {
        const result = await apiFetch({
          path: "/vidtoarticle/v1/admin/connect",
          method: "POST",
        });

        if (result.success && result.url) {
          // Redirect to VidToArticle OAuth
          // The PHP OAuth handler will redirect back with connection established
          window.location.href = result.url;
        } else {
          throw new Error(result.error || "Failed to initiate connection");
        }
      } catch (error) {
        addNotice(
          "error",
          __("Connection failed: ", "vidtoarticle-publisher") + error.message
        );
        setConnecting(false);
      }
    };

    /**
     * Disconnect from VidToArticle.com
     */
    const handleDisconnect = async () => {
      if (
        !confirm(
          __(
            "Are you sure you want to disconnect? This will stop all automatic publishing.",
            "vidtoarticle-publisher"
          )
        )
      ) {
        return;
      }

      setDisconnecting(true);
      try {
        await apiFetch({
          path: "/vidtoarticle/v1/admin/disconnect",
          method: "POST",
        });

        addNotice(
          "success",
          __("Disconnected successfully", "vidtoarticle-publisher")
        );
        await refreshConnection();
      } catch (error) {
        addNotice(
          "error",
          __("Disconnect failed: ", "vidtoarticle-publisher") + error.message
        );
      }
      setDisconnecting(false);
    };

    // ========================================================================
    // Connected State UI
    // ========================================================================
    const renderConnectedState = () =>
      el(
        Fragment,
        null,
        // Success notice
        el(
          Notice,
          { status: "success", isDismissible: false },
          __("✓ Connected to VidToArticle.com", "vidtoarticle-publisher")
        ),

        // Primary CTA - Manage on VidToArticle.com
        el(
          "div",
          { className: "connection-cta", style: { marginTop: "24px", marginBottom: "24px" } },
          el(
            "div",
            {
              style: {
                background: "linear-gradient(135deg, #f0f6fc 0%, #e6f0f8 100%)",
                border: "1px solid #c3d9ed",
                borderRadius: "8px",
                padding: "20px",
                textAlign: "center",
              },
            },
            el(
              "p",
              {
                style: {
                  margin: "0 0 16px 0",
                  fontSize: "15px",
                  color: "#1e3a5f",
                },
              },
              __(
                "Configure your YouTube sources, article styles, and publishing settings on VidToArticle.com",
                "vidtoarticle-publisher"
              )
            ),
            el(
              Button,
              {
                variant: "primary",
                href: DASHBOARD_URL,
                target: "_blank",
                rel: "noopener noreferrer",
                style: { fontSize: "14px" },
              },
              el("span", { className: "dashicons dashicons-external", style: { marginRight: "8px" } }),
              __("Open Dashboard →", "vidtoarticle-publisher")
            )
          )
        ),

        // Connection details
        el(
          "div",
          { className: "connection-info" },
          el("h3", null, __("Connection Details", "vidtoarticle-publisher")),
          el(
            "table",
            { className: "widefat" },
            el(
              "tbody",
              null,
              el(
                "tr",
                null,
                el("th", null, __("Status", "vidtoarticle-publisher")),
                el(
                  "td",
                  null,
                  el(
                    "span",
                    {
                      style: {
                        color: connectionStatus.health_status === "healthy" ? "#00a32a" : "#dba617",
                        fontWeight: "500",
                      },
                    },
                    connectionStatus.health_status === "healthy"
                      ? __("Healthy", "vidtoarticle-publisher")
                      : connectionStatus.health_status || __("Connected", "vidtoarticle-publisher")
                  )
                )
              ),
              el(
                "tr",
                null,
                el("th", null, __("Site", "vidtoarticle-publisher")),
                el("td", null, connectionStatus.domain || window.vidToArticleAdmin?.siteUrl || "-")
              ),
              connectionStatus.api_key_id &&
                el(
                  "tr",
                  null,
                  el("th", null, __("API Key", "vidtoarticle-publisher")),
                  el("td", null, el("code", null, connectionStatus.api_key_id))
                ),
              connectionStatus.sources_count !== undefined &&
                el(
                  "tr",
                  null,
                  el("th", null, __("Active Sources", "vidtoarticle-publisher")),
                  el("td", null, connectionStatus.sources_count)
                ),
              connectionStatus.last_activity &&
                el(
                  "tr",
                  null,
                  el("th", null, __("Last Activity", "vidtoarticle-publisher")),
                  el(
                    "td",
                    null,
                    new Date(connectionStatus.last_activity).toLocaleString()
                  )
                )
            )
          )
        ),

        // Actions
        el(
          "div",
          { className: "connection-actions", style: { marginTop: "20px" } },
          el(
            Button,
            {
              variant: "secondary",
              onClick: refreshConnection,
            },
            __("Refresh Status", "vidtoarticle-publisher")
          ),
          " ",
          el(
            Button,
            {
              variant: "tertiary",
              isDestructive: true,
              onClick: handleDisconnect,
              isBusy: disconnecting,
            },
            __("Disconnect", "vidtoarticle-publisher")
          )
        )
      );

    // ========================================================================
    // Disconnected State UI
    // ========================================================================
    const renderDisconnectedState = () =>
      el(
        Fragment,
        null,
        el(
          Notice,
          { status: "warning", isDismissible: false },
          __("Not connected to VidToArticle.com", "vidtoarticle-publisher")
        ),
        el(
          "div",
          { className: "connection-setup" },
          el(
            "h3",
            null,
            __("Connect Your WordPress Site", "vidtoarticle-publisher")
          ),
          el(
            "p",
            null,
            __(
              "Connect to VidToArticle.com to automatically generate and publish articles from YouTube videos.",
              "vidtoarticle-publisher"
            )
          ),
          el(
            "ol",
            null,
            el(
              "li",
              null,
              __("Click the Connect button below", "vidtoarticle-publisher")
            ),
            el(
              "li",
              null,
              __(
                "Sign in to your VidToArticle.com account (or create one)",
                "vidtoarticle-publisher"
              )
            ),
            el(
              "li",
              null,
              __("Authorize this WordPress site", "vidtoarticle-publisher")
            ),
            el(
              "li",
              null,
              __(
                "You'll be redirected back here automatically",
                "vidtoarticle-publisher"
              )
            )
          ),
          el(
            Button,
            {
              variant: "primary",
              onClick: handleConnect,
              isBusy: connecting,
              style: { marginTop: "8px" },
            },
            __("Connect to VidToArticle", "vidtoarticle-publisher")
          )
        )
      );

    // ========================================================================
    // Main Render
    // ========================================================================
    return el(
      Card,
      null,
      el(
        CardBody,
        null,
        connectionStatus.connected
          ? renderConnectedState()
          : renderDisconnectedState()
      )
    );
  }

  // Register component in global registry
  window.VidToArticleTabs = window.VidToArticleTabs || {};
  window.VidToArticleTabs.ConnectionTab = ConnectionTab;
})();
