(function () {
  "use strict";

  const { createElement: el, useEffect, useState, useCallback } = wp.element;
  const { Notice, Spinner, Button, SelectControl, TabPanel, CheckboxControl } = wp.components;
  const { __ } = wp.i18n;
  const apiFetch = wp.apiFetch;

  /**
   * Log level badge component
   */
  function LogLevelBadge({ level }) {
    const colors = {
      error: { bg: "#fee2e2", color: "#dc2626", text: "ERROR" },
      warning: { bg: "#fef3c7", color: "#d97706", text: "WARNING" },
      info: { bg: "#dbeafe", color: "#2563eb", text: "INFO" },
      debug: { bg: "#f3f4f6", color: "#6b7280", text: "DEBUG" },
    };

    const style = colors[level] || colors.info;

    return el(
      "span",
      {
        style: {
          display: "inline-block",
          padding: "2px 8px",
          borderRadius: "4px",
          fontSize: "11px",
          fontWeight: "600",
          backgroundColor: style.bg,
          color: style.color,
          textTransform: "uppercase",
        },
      },
      style.text
    );
  }

  /**
   * Log entry row component
   */
  function LogRow({ item, showType }) {
    const [expanded, setExpanded] = useState(false);
    const hasContext = item.context && Object.keys(item.context).length > 0;

    // Parse level from action if not already present
    let level = item.level || "info";
    if (!item.level && item.action) {
      if (item.action.includes("error")) level = "error";
      else if (item.action.includes("warning")) level = "warning";
      else if (item.action.includes("debug")) level = "debug";
    }

    return el(
      "tr",
      { style: { verticalAlign: "top" } },
      el("td", { style: { whiteSpace: "nowrap", width: "150px" } },
        new Date(item.created_at || item.createdAt).toLocaleString()
      ),
      el("td", { style: { width: "80px" } },
        el(LogLevelBadge, { level })
      ),
      showType && el("td", { style: { width: "80px" } },
        el("code", { style: { fontSize: "11px" } }, item.type || "plugin")
      ),
      el("td", { style: { width: "140px" } },
        el("code", { style: { fontSize: "12px" } }, item.action)
      ),
      el(
        "td",
        null,
        el("div", null, item.message),
        hasContext &&
          el(
            "div",
            { style: { marginTop: "8px" } },
            el(
              Button,
              {
                variant: "link",
                onClick: () => setExpanded(!expanded),
                style: { fontSize: "12px", padding: 0, height: "auto" },
              },
              expanded ? __("Hide Details", "vidtoarticle-publisher") : __("Show Details", "vidtoarticle-publisher")
            ),
            expanded &&
              el(
                "pre",
                {
                  style: {
                    background: "#f5f5f5",
                    padding: "8px",
                    borderRadius: "4px",
                    fontSize: "11px",
                    overflow: "auto",
                    marginTop: "8px",
                    maxHeight: "200px",
                  },
                },
                JSON.stringify(item.context, null, 2)
              )
          )
      )
    );
  }

  /**
   * Plugin logs section (local WordPress logs)
   */
  function PluginLogs({ addNotice, levelFilter }) {
    const [logs, setLogs] = useState([]);
    const [loading, setLoading] = useState(true);

    const loadLogs = useCallback(async () => {
      setLoading(true);
      try {
        const result = await apiFetch({
          path: "/vidtoarticle/v1/admin/activity",
        });
        let activity = result.activity || [];

        // Apply level filter
        if (levelFilter && levelFilter !== "all") {
          activity = activity.filter((item) => {
            const level = item.action?.includes("error")
              ? "error"
              : item.action?.includes("warning")
                ? "warning"
                : item.action?.includes("debug")
                  ? "debug"
                  : "info";
            return level === levelFilter;
          });
        }

        setLogs(activity);
      } catch (error) {
        addNotice("error", __("Failed to load plugin logs", "vidtoarticle-publisher"));
      }
      setLoading(false);
    }, [addNotice, levelFilter]);

    useEffect(() => {
      loadLogs();
    }, [loadLogs]);

    if (loading) {
      return el(
        "div",
        { style: { textAlign: "center", padding: "20px" } },
        el(Spinner)
      );
    }

    if (logs.length === 0) {
      return el(
        Notice,
        { status: "info", isDismissible: false },
        __("No plugin logs found", "vidtoarticle-publisher")
      );
    }

    return el(
      "table",
      { className: "wp-list-table widefat fixed striped", style: { tableLayout: "auto" } },
      el(
        "thead",
        null,
        el(
          "tr",
          null,
          el("th", { style: { width: "150px" } }, __("Time", "vidtoarticle-publisher")),
          el("th", { style: { width: "80px" } }, __("Level", "vidtoarticle-publisher")),
          el("th", { style: { width: "140px" } }, __("Action", "vidtoarticle-publisher")),
          el("th", null, __("Message", "vidtoarticle-publisher"))
        )
      ),
      el(
        "tbody",
        null,
        logs.map((item, index) =>
          el(LogRow, { key: index, item, showType: false })
        )
      )
    );
  }

  /**
   * Backend logs section (logs from VidToArticle.com)
   */
  function BackendLogs({ addNotice, levelFilter }) {
    const [logs, setLogs] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);

    const loadLogs = useCallback(async () => {
      setLoading(true);
      setError(null);
      try {
        const result = await apiFetch({
          path: `/vidtoarticle/v1/admin/backend-logs?level=${levelFilter || "all"}&limit=100`,
        });

        if (!result.success) {
          setError(result.error || __("Failed to fetch backend logs", "vidtoarticle-publisher"));
          setLogs([]);
        } else {
          setLogs(result.logs || []);
        }
      } catch (err) {
        setError(err.message || __("Failed to load backend logs", "vidtoarticle-publisher"));
        setLogs([]);
      }
      setLoading(false);
    }, [addNotice, levelFilter]);

    useEffect(() => {
      loadLogs();
    }, [loadLogs]);

    if (loading) {
      return el(
        "div",
        { style: { textAlign: "center", padding: "20px" } },
        el(Spinner),
        el("p", { style: { marginTop: "10px", color: "#666" } },
          __("Fetching logs from VidToArticle.com...", "vidtoarticle-publisher")
        )
      );
    }

    if (error) {
      return el(
        Notice,
        { status: "warning", isDismissible: false },
        el("strong", null, __("Could not fetch backend logs: ", "vidtoarticle-publisher")),
        error
      );
    }

    if (logs.length === 0) {
      return el(
        Notice,
        { status: "info", isDismissible: false },
        __("No backend logs found. Logs will appear here when articles are processed or webhook deliveries occur.", "vidtoarticle-publisher")
      );
    }

    return el(
      "table",
      { className: "wp-list-table widefat fixed striped", style: { tableLayout: "auto" } },
      el(
        "thead",
        null,
        el(
          "tr",
          null,
          el("th", { style: { width: "150px" } }, __("Time", "vidtoarticle-publisher")),
          el("th", { style: { width: "80px" } }, __("Level", "vidtoarticle-publisher")),
          el("th", { style: { width: "80px" } }, __("Type", "vidtoarticle-publisher")),
          el("th", { style: { width: "140px" } }, __("Action", "vidtoarticle-publisher")),
          el("th", null, __("Message", "vidtoarticle-publisher"))
        )
      ),
      el(
        "tbody",
        null,
        logs.map((item, index) =>
          el(LogRow, { key: index, item, showType: true })
        )
      )
    );
  }

  /**
   * Main Activity Tab (now "Logs" tab)
   */
  function ActivityTab({ addNotice }) {
    const [levelFilter, setLevelFilter] = useState("all");
    const [activeTab, setActiveTab] = useState("backend");
    const [autoRefresh, setAutoRefresh] = useState(false);
    const [refreshKey, setRefreshKey] = useState(0);

    // Auto-refresh effect
    useEffect(() => {
      if (!autoRefresh) return;

      const interval = setInterval(() => {
        setRefreshKey((k) => k + 1);
      }, 30000); // 30 seconds

      return () => clearInterval(interval);
    }, [autoRefresh]);

    const tabContent = {
      backend: el(BackendLogs, { key: refreshKey, addNotice, levelFilter }),
      plugin: el(PluginLogs, { key: refreshKey, addNotice, levelFilter }),
    };

    return el(
      "div",
      { className: "vidtoarticle-logs-container" },
      // Header with controls
      el(
        "div",
        {
          style: {
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
            marginBottom: "20px",
            flexWrap: "wrap",
            gap: "10px",
          },
        },
        el(
          "div",
          { style: { display: "flex", alignItems: "center", gap: "15px" } },
          el(SelectControl, {
            label: __("Filter by Level", "vidtoarticle-publisher"),
            value: levelFilter,
            options: [
              { label: __("All Levels", "vidtoarticle-publisher"), value: "all" },
              { label: __("Errors Only", "vidtoarticle-publisher"), value: "error" },
              { label: __("Warnings", "vidtoarticle-publisher"), value: "warning" },
              { label: __("Info", "vidtoarticle-publisher"), value: "info" },
              { label: __("Debug", "vidtoarticle-publisher"), value: "debug" },
            ],
            onChange: setLevelFilter,
            __nextHasNoMarginBottom: true,
            style: { minWidth: "150px" },
          }),
          el(CheckboxControl, {
            label: __("Auto-refresh (30s)", "vidtoarticle-publisher"),
            checked: autoRefresh,
            onChange: setAutoRefresh,
            __nextHasNoMarginBottom: true,
          })
        ),
        el(
          Button,
          {
            variant: "secondary",
            onClick: () => setRefreshKey((k) => k + 1),
            style: { marginLeft: "auto" },
          },
          __("Refresh", "vidtoarticle-publisher")
        )
      ),

      // Tab navigation
      el(
        "div",
        {
          style: {
            display: "flex",
            borderBottom: "1px solid #c3c4c7",
            marginBottom: "20px",
          },
        },
        el(
          "button",
          {
            type: "button",
            className: `nav-tab ${activeTab === "backend" ? "nav-tab-active" : ""}`,
            onClick: () => setActiveTab("backend"),
            style: {
              padding: "10px 20px",
              cursor: "pointer",
              border: "1px solid #c3c4c7",
              borderBottom: activeTab === "backend" ? "1px solid #fff" : "none",
              marginBottom: "-1px",
              background: activeTab === "backend" ? "#fff" : "#f0f0f1",
              fontWeight: activeTab === "backend" ? "600" : "400",
            },
          },
          __("Backend Logs", "vidtoarticle-publisher"),
          " ",
          el("span", {
            style: {
              fontSize: "10px",
              color: "#666",
              marginLeft: "5px",
            },
          }, "(VidToArticle.com)")
        ),
        el(
          "button",
          {
            type: "button",
            className: `nav-tab ${activeTab === "plugin" ? "nav-tab-active" : ""}`,
            onClick: () => setActiveTab("plugin"),
            style: {
              padding: "10px 20px",
              cursor: "pointer",
              border: "1px solid #c3c4c7",
              borderBottom: activeTab === "plugin" ? "1px solid #fff" : "none",
              marginBottom: "-1px",
              background: activeTab === "plugin" ? "#fff" : "#f0f0f1",
              fontWeight: activeTab === "plugin" ? "600" : "400",
            },
          },
          __("Plugin Logs", "vidtoarticle-publisher"),
          " ",
          el("span", {
            style: {
              fontSize: "10px",
              color: "#666",
              marginLeft: "5px",
            },
          }, "(Local)")
        )
      ),

      // Tab content
      tabContent[activeTab]
    );
  }

  window.VidToArticleTabs = window.VidToArticleTabs || {};
  window.VidToArticleTabs.ActivityTab = ActivityTab;
})();
