/**
 * VidToArticle Admin Interface - React Application
 *
 * Simplified plugin interface focusing on:
 * 1. Connection management (OAuth with VidToArticle.com)
 * 2. Activity logging (for transparency and debugging)
 *
 * All source configuration and queue management is handled on VidToArticle.com
 * This plugin acts as a lightweight connector and publisher.
 *
 * @package VidToArticle_Publisher
 * @since 1.3.0
 */

(function () {
  "use strict";

  // ============================================================================
  // WordPress Dependencies
  // ============================================================================
  const { createElement: el, Fragment, useState, useEffect } = wp.element;
  const { TabPanel, Notice, Spinner } = wp.components;
  const { __ } = wp.i18n;
  const apiFetch = wp.apiFetch;

  // ============================================================================
  // Module Registry - Lazy loaded components
  // ============================================================================
  const tabsRegistry = (window.VidToArticleTabs =
    window.VidToArticleTabs || {});

  // ============================================================================
  // Constants
  // ============================================================================
  const MODULE_CHECK_MAX = 3;
  const MODULE_CHECK_INTERVAL_MS = 150;
  const NOTICE_AUTO_DISMISS_MS = 5000;

  /**
   * Tab configuration - simplified to Connection and Activity only
   * Sources and Queue are managed on VidToArticle.com
   */
  const TAB_CONFIG = {
    connection: {
      componentName: "ConnectionTab",
      title: __("Connection", "vidtoarticle-publisher"),
      icon: "admin-links",
    },
    activity: {
      componentName: "ActivityTab",
      title: __("Activity Log", "vidtoarticle-publisher"),
      icon: "list-view",
    },
  };

  // ============================================================================
  // Utility Functions
  // ============================================================================

  /**
   * Check which required tab modules are missing
   * @returns {string[]} Array of missing module names
   */
  const getMissingModules = () => {
    return Object.values(TAB_CONFIG)
      .map((config) => config.componentName)
      .filter((name) => !tabsRegistry[name]);
  };

  /**
   * Generate unique ID for notices
   * @returns {number} Unique timestamp-based ID
   */
  const generateNoticeId = () => Date.now();

  // ============================================================================
  // API Configuration
  // ============================================================================
  if (window.vidToArticleAdmin?.nonce) {
    apiFetch.use(
      apiFetch.createNonceMiddleware(window.vidToArticleAdmin.nonce)
    );
  }

  // ============================================================================
  // Sub-Components
  // ============================================================================

  /**
   * Loading spinner with message
   */
  const LoadingState = ({ message }) =>
    el(
      "div",
      { className: "vidtoarticle-admin-loading" },
      el(Spinner),
      el("p", null, message)
    );

  /**
   * Error notice for missing modules
   */
  const MissingModulesError = ({ modules }) =>
    el(
      "div",
      { className: "vidtoarticle-admin-loading" },
      el(
        Notice,
        { status: "error", isDismissible: false },
        el(
          "p",
          null,
          __(
            "Some admin components failed to initialize.",
            "vidtoarticle-publisher"
          )
        ),
        el(
          "ul",
          null,
          modules.map((name) =>
            el("li", { key: name }, `${name} (${__("chunk", "vidtoarticle-publisher")})`)
          )
        ),
        el(
          "p",
          null,
          __(
            "Please refresh the page to retry loading.",
            "vidtoarticle-publisher"
          )
        )
      )
    );

  /**
   * Connection status indicator in header
   */
  const ConnectionIndicator = ({ connected }) =>
    el(
      "div",
      { className: "vidtoarticle-connection-indicator" },
      el("span", {
        className: `status-dot ${connected ? "status-connected" : "status-disconnected"}`,
      }),
      el(
        "span",
        { className: "status-text" },
        connected
          ? __("Connected", "vidtoarticle-publisher")
          : __("Not Connected", "vidtoarticle-publisher")
      )
    );

  /**
   * Notices container for user feedback
   */
  const NoticesContainer = ({ notices, onDismiss }) =>
    el(
      "div",
      { className: "vidtoarticle-admin-notices" },
      notices.map((notice) =>
        el(
          Notice,
          {
            key: notice.id,
            status: notice.type,
            isDismissible: true,
            onRemove: () => onDismiss(notice.id),
          },
          notice.message
        )
      )
    );

  // ============================================================================
  // Main App Component
  // ============================================================================

  function App() {
    // State management
    const [loading, setLoading] = useState(true);
    const [loadError, setLoadError] = useState(null);
    const [connectionStatus, setConnectionStatus] = useState(null);
    const [notices, setNotices] = useState([]);
    const [missingModules, setMissingModules] = useState(getMissingModules());
    const [moduleChecks, setModuleChecks] = useState(0);

    // ========================================================================
    // Effects
    // ========================================================================

    // Initial data load
    useEffect(() => {
      loadInitialData();
    }, []);

    // Module availability polling
    useEffect(() => {
      if (missingModules.length === 0 || moduleChecks >= MODULE_CHECK_MAX) {
        return;
      }
      const timer = setTimeout(() => {
        setMissingModules(getMissingModules());
        setModuleChecks((prev) => prev + 1);
      }, MODULE_CHECK_INTERVAL_MS);
      return () => clearTimeout(timer);
    }, [missingModules, moduleChecks]);

    // ========================================================================
    // Data Loading
    // ========================================================================

    const loadInitialData = async () => {
      try {
        const connStatus = await apiFetch({
          path: "/vidtoarticle/v1/admin/connection",
        });
        setConnectionStatus(connStatus);
        setLoading(false);
        setLoadError(null);
      } catch (error) {
        console.error("VidToArticle API Error:", error);
        setLoadError(error.message || __("Failed to connect to REST API", "vidtoarticle-publisher"));
        setLoading(false);
      }
    };

    // ========================================================================
    // Notice Management
    // ========================================================================

    const addNotice = (type, message) => {
      const id = generateNoticeId();
      setNotices((prev) => [...prev, { id, type, message }]);
      setTimeout(() => {
        setNotices((prev) => prev.filter((n) => n.id !== id));
      }, NOTICE_AUTO_DISMISS_MS);
    };

    const dismissNotice = (id) => {
      setNotices((prev) => prev.filter((n) => n.id !== id));
    };

    // ========================================================================
    // Connection Management
    // ========================================================================

    const refreshConnection = async () => {
      try {
        const connStatus = await apiFetch({
          path: "/vidtoarticle/v1/admin/connection",
        });
        setConnectionStatus(connStatus);
      } catch (error) {
        addNotice(
          "error",
          __("Failed to refresh connection status", "vidtoarticle-publisher")
        );
      }
    };

    // ========================================================================
    // Render Logic
    // ========================================================================

    // Still loading modules
    if (missingModules.length > 0) {
      if (moduleChecks < MODULE_CHECK_MAX) {
        return el(LoadingState, {
          message: __("Loading interface assets...", "vidtoarticle-publisher"),
        });
      }
      return el(MissingModulesError, { modules: missingModules });
    }

    // Still loading data
    if (loading) {
      return el(LoadingState, {
        message: __("Loading...", "vidtoarticle-publisher"),
      });
    }

    // Build tabs array from config
    const tabs = Object.entries(TAB_CONFIG).map(([name, config]) => ({
      name,
      title: config.title,
      icon: config.icon,
    }));

    // Tab content renderer
    const renderTabContent = (tabName) => {
      const config = TAB_CONFIG[tabName];
      if (!config) return null;

      const TabComponent = tabsRegistry[config.componentName];
      if (!TabComponent) {
        return el(
          Notice,
          { status: "error", isDismissible: false },
          __("Component failed to load.", "vidtoarticle-publisher") +
            ` (${config.componentName})`
        );
      }

      // Pass appropriate props based on tab
      const tabProps = {
        addNotice,
        ...(tabName === "connection" && {
          connectionStatus,
          refreshConnection,
        }),
      };

      return el(TabComponent, tabProps);
    };

    // Main render
    return el(
      Fragment,
      null,
      // Header with connection indicator
      el(
        "div",
        { className: "vidtoarticle-admin-header" },
        el(
          "div",
          { className: "vidtoarticle-header-content" },
          el(
            "h1",
            null,
            el("span", { className: "dashicons dashicons-video-alt3" }),
            __("VidToArticle", "vidtoarticle-publisher")
          ),
          connectionStatus &&
            el(ConnectionIndicator, { connected: connectionStatus.connected })
        ),
        el(
          "p",
          { className: "vidtoarticle-header-description" },
          __("Automatically publish articles from YouTube videos. Configure sources and settings on ", "vidtoarticle-publisher"),
          el(
            "a",
            {
              href: "https://www.vidtoarticle.com/wordpress-connect",
              target: "_blank",
              rel: "noopener noreferrer",
            },
            "VidToArticle.com"
          ),
          "."
        )
      ),

      // Error display
      loadError &&
        el(
          "div",
          { className: "vidtoarticle-admin-errors" },
          el(Notice, { status: "error", isDismissible: true }, loadError)
        ),

      // Notices
      notices.length > 0 &&
        el(NoticesContainer, { notices, onDismiss: dismissNotice }),

      // Tab panel - simplified to Connection + Activity
      el(
        TabPanel,
        {
          className: "vidtoarticle-admin-tabs",
          activeClass: "is-active",
          tabs,
        },
        (tab) => renderTabContent(tab.name)
      )
    );
  }

  // ============================================================================
  // App Initialization
  // ============================================================================
  const root = document.getElementById("vidtoarticle-admin-root");
  if (root) {
    wp.element.render(el(App), root);
  }
})();
